<?php
/**
 * app/Views/matrices2/grid.php
 *
 * Vista tipo “Excel” (UI moderna + súper compacta):
 * - Carga total (hasta ~50k filas) por chunks adaptativos (SIN CAMBIOS)
 * - Columnas base + dinámicas (sin duplicados)
 * - Solo edita columnas permitidas (no fórmula)
 * - Recalcula fila al guardar
 * - Filtros por columna en cabecera
 * - Mover columnas
 * - Ocultar/mostrar columnas (columna completa)
 * - Ultra-compacto 16px
 */

$matId   = (int)($mat['mat_id'] ?? 0);
$matTipo = esc($mat['mat_tipo'] ?? '');
$matNom  = esc($mat['mat_nombre'] ?? '');

/** Campos base fijos */
$BASE_PROPS = ['pro_codigo','pro_descripcion','pro_familia','pro_categorizacion'];

/** DynCols desde $campos sin colisiones/duplicados */
$tmp = array_values(array_filter(array_map(function($c){
  if ((int)($c['mcr_visible'] ?? 1) !== 1) return null;
  return [
    'name'       => $c['cam_nombre'] ?? '',
    'title'      => ($c['mcr_titulo'] ?: ($c['cam_titulo'] ?: ($c['cam_nombre'] ?? ''))),
    'tipo'       => $c['cam_tipo'] ?? 'text',
    'origen'     => $c['cam_origen'] ?? 'lectura',
    'hasSource'  => !empty($c['cam_source']),
    'editable'   => (bool)($c['editable'] ?? false),
  ];
}, $campos ?? [])));

$seen  = [];
$dynCols = [];
foreach ($tmp as $col) {
  $name = (string)($col['name'] ?? '');
  if ($name === '' || in_array($name, $BASE_PROPS, true)) continue;
  if (isset($seen[$name])) continue;
  $seen[$name] = true;
  $dynCols[] = $col;
}
?>
<div class="container-fluid" style="margin-top:80px; --xl-blue:#1976d2; --xl-blue-50:#eef5ff; --xl-blue-100:#e2ecfb; --xl-border:#c7dbf7; --xl-grid:#e6ebf5;"> 
<div class="d-flex justify-content-between align-items-center mb-3"> 
<div> <h2 class="mb-0" style="font-weight:700;color:var(--xl-blue);display:flex;align-items:center;gap:10px"> 
<span style="display:inline-block;width:14px;height:24px;background:linear-gradient(180deg,var(--xl-blue),#42a5f5);border-radius:2px"></span> 
<?= $matNom ?> </h2> 

</div> 
<!-- === Toolbar de acciones (una sola franja) === -->
<div class="mat-actionbar">
  <div class="mat-actions-left">
    <button id="btn-export" class="btn btn-ghost btn-sm">
      <span class="glyphicon glyphicon-export"></span> Exportar CSV
    </button>
    
    <a class="btn btn-ghost btn-sm" href="<?= site_url('matrices2'); ?>">
      <span class="glyphicon glyphicon-menu-left"></span> Volver
    </a>
  </div>

  <div class="mat-actions-right">
    <button class="btn btn-primary btn-sm btn-pill" data-toggle="modal" data-target="#mdlDist">
      <span class="glyphicon glyphicon-upload"></span> Cargar precios distribución
    </button>
    <button class="btn btn-info btn-sm btn-pill" data-toggle="modal" data-target="#mdlPDV">
      <span class="glyphicon glyphicon-upload"></span> Cargar precios PDV
    </button>
    <button class="btn btn-warning btn-sm btn-pill" data-toggle="modal" data-target="#mdlCostos">
      <span class="glyphicon glyphicon-upload"></span> Cargar costos
    </button>
  </div>
</div>
<style>
/* ===== Toolbar de acciones ===== */
.mat-actionbar{
  display:flex; align-items:center; justify-content:space-between;
  gap:12px; padding:10px 12px;
  background:linear-gradient(180deg, var(--xl-blue-50), var(--xl-blue-100));
  border:1px solid var(--xl-border); border-radius:10px;
  box-shadow:0 1px 0 rgba(15,23,42,.06); margin:8px 0 10px;
}
.mat-actions-left, .mat-actions-right{
  display:flex; align-items:center; gap:8px; flex-wrap:wrap;
}

/* Botón “ghost” para acciones secundarias (Exportar/Volver) */
.btn-ghost{
  background:#fff; color:#0d2a44; border:1px solid var(--xl-border);
  border-radius:10px; font-weight:600;
}
.btn-ghost:hover{
  border-color: var(--xl-blue);
  box-shadow:0 3px 10px rgba(25,118,210,.12);
}

/* Botones principales */
.btn-pill{ border-radius:999px; font-weight:700; padding:6px 12px; }
.btn-sm{ line-height:1.25; height:32px; display:inline-flex; align-items:center; gap:6px; }

/* Responsive: si no cabe, se quiebra a 2 líneas ordenadas */
@media (max-width: 900px){
  .mat-actionbar{ flex-wrap:wrap; }
  .mat-actions-left{ width:100%; order:1; }
  .mat-actions-right{ width:100%; order:2; justify-content:flex-start; }
}

/* ===== MODAL · Contenedor ===== */
  .hq-modal .modal-dialog{ max-width:760px; width:96%; }
  .hq-modal .modal-content{
    border:1px solid var(--xl07-border-h);
    border-radius:12px;
    background:#fff;
    box-shadow:0 12px 36px rgba(13, 31, 64, .18);
    overflow:hidden;
  }

  /* ===== MODAL · Header con cinta y gradiente sistema ===== */
  .hq-modal .modal-header{
    padding:14px 18px;
    border:0;
    background:linear-gradient(180deg, var(--xl-blue-50), var(--xl-blue-100));
    position:relative;
  }
  .hq-modal .modal-header::after{
    content:"";
    position:absolute; left:0; right:0; bottom:0; height:1px;
    background:var(--xl07-border-c);
  }
  .hq-modal .modal-title{
    margin:0;
    font-weight:700;
    color:#0d2a44; /* igual a cabeceras Excel 2007 */
    letter-spacing:.2px;
  }
  .hq-modal .close{ opacity:.6; text-shadow:none; }
  .hq-modal .close:hover{ opacity:.9; }

  /* ===== MODAL · Body ===== */
  .hq-modal .modal-body{
    padding:20px 18px 16px;
    background:#fff;
  }

  /* ===== Campos en 2 columnas ===== */
  .hq-row{
    display:flex; align-items:center; gap:16px;
    margin-bottom:16px;
  }
  .hq-chip{
    min-width:200px;
    padding:8px 12px;
    border-radius:10px;
    border:1px solid var(--xl07-border-h);
    background:linear-gradient(180deg, var(--xl-blue-50), #ffffff);
    color:#0d2a44;
    font-weight:700;
  }
  .hq-chip .tag{
    display:inline-block; background:#fff; border:1px solid var(--xl07-border-c);
    padding:2px 8px; border-radius:6px; font-weight:600;
  }
  .hq-field{ flex:1; }
  .hq-field .form-control{
    height:36px; border-radius:8px; box-shadow:none;
    border:1px solid var(--xl07-border-c); transition:border-color .15s, box-shadow .15s;
    font-family: Calibri, Roboto, system-ui, -apple-system, "Segoe UI", Arial, sans-serif;
  }
  .hq-field .form-control:focus{
    border-color: var(--xl-blue);
    box-shadow:0 0 0 3px rgba(25,118,210,.15);
  }

  /* Fecha con ícono */
  .hq-date-wrap{ position:relative; }
  .hq-date-wrap .glyphicon{
    position:absolute; right:10px; top:9px; opacity:.55;
  }

  /* Promoción (columna lateral) */
  .hq-side{
    width:150px; padding-left:4px;
  }
  .hq-side label{ font-weight:600; color:#0f2942; }

  /* Mensaje confirmación */
  .hq-inline-hint{
    text-align:center; color:#425466; font-size:13px;
    background: #f7fbff; border:1px dashed var(--xl07-border-c);
    padding:10px 12px; border-radius:10px; margin:8px 0 6px;
  }
  .hq-inline-hint strong{ color:#0d2a44; }

  /* ===== Botones ===== */
  .hq-actions{ display:flex; justify-content:center; gap:14px; margin-top:10px; }
  .btn-hq{
    min-width:150px; font-weight:700; border-radius:12px;
    border:1px solid var(--xl07-border-h);
    background: linear-gradient(180deg, #ffffff, var(--xl-blue-50));
    color:#0d2a44;
  }
  .btn-hq:hover{
    border-color: var(--xl-blue);
    box-shadow:0 4px 14px rgba(25,118,210,.12);
  }
  .btn-hq-primary{
    background: linear-gradient(180deg, var(--xl-blue), #42a5f5);
    color:#fff; border:1px solid var(--xl-blue);
  }
  .btn-hq-primary:hover{
    filter: brightness(.98);
    box-shadow:0 6px 16px rgba(25,118,210,.25);
  }

  /* ===== Responsive ===== */
  @media (max-width:640px){
    .hq-row{ display:block; }
    .hq-chip{ margin-bottom:8px; min-width:unset; }
    .hq-side{ width:auto; padding-left:0; margin-top:8px; }
  }
</style>

<!-- ===== MODAL: Cargar precios distribución ===== -->
<div id="mdlDist" class="modal fade hq-modal" tabindex="-1" role="dialog" aria-hidden="true" data-count="50">
  <div class="modal-dialog"><div class="modal-content">
    <div class="modal-header">
      <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
      <h4 class="modal-title">HQ — Cargar precios distribución</h4>
    </div>
    <div class="modal-body">
      <form>
        <div class="hq-row">
          <div class="hq-chip purple">Hasta cuando:</div>
          <div class="hq-field">
            <div class="hq-date-wrap">
              <input type="date" class="form-control" placeholder="AAAA-MM-DD">
              <span class="glyphicon glyphicon-calendar"></span>
            </div>
          </div>
          <div style="width:140px; padding-left:14px">
            <label style="font-weight:600; margin:0 0 6px; display:block">Promoción</label>
            <div class="checkbox" style="margin:0"><label style="font-weight:600"><input type="checkbox"></label></div>
          </div>
        </div>

        <div class="hq-row">
          <div class="hq-chip yellow">Motivo de cambio:</div>
          <div class="hq-field"><input type="text" class="form-control" placeholder="Describe el motivo…"></div>
        </div>

        <div class="hq-row">
          <div class="hq-chip peach"><span style="display:inline-block;background:#fff;padding:2px 8px;border-radius:6px;border:1px solid #e2e8f0">Usuario:</span></div>
          <div class="hq-field"><input type="text" class="form-control" value="<?= esc($usuario ?? 'DARONDON') ?>" readonly></div>
        </div>

        <div class="hq-inline-hint">
          Va a realizar cambios en <strong class="hq-count">50</strong> códigos, ¿estás seguro?
        </div>

        <div class="hq-actions">
          <button type="button" class="btn btn-default" data-dismiss="modal">Cancelar</button>
          <button type="button" class="btn">Aceptar</button>
        </div>
      </form>
    </div>
  </div></div>
</div>

<!-- ===== MODAL: Cargar precios PDV ===== -->
<div id="mdlPDV" class="modal fade hq-modal" tabindex="-1" role="dialog" aria-hidden="true" data-count="50">
  <div class="modal-dialog"><div class="modal-content">
    <div class="modal-header">
      <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
      <h4 class="modal-title">HQ — Cargar precios PDV</h4>
    </div>
    <div class="modal-body">
      <form>
        <div class="hq-row">
          <div class="hq-chip purple">Hasta cuando:</div>
          <div class="hq-field">
            <div class="hq-date-wrap">
              <input type="date" class="form-control">
              <span class="glyphicon glyphicon-calendar"></span>
            </div>
          </div>
          <div style="width:140px; padding-left:14px">
            <label style="font-weight:600; margin:0 0 6px; display:block">Promoción</label>
            <div class="checkbox" style="margin:0"><label style="font-weight:600"><input type="checkbox"></label></div>
          </div>
        </div>

        <div class="hq-row">
          <div class="hq-chip yellow">Motivo de cambio:</div>
          <div class="hq-field"><input type="text" class="form-control"></div>
        </div>

        <div class="hq-row">
          <div class="hq-chip peach"><span style="display:inline-block;background:#fff;padding:2px 8px;border-radius:6px;border:1px solid #e2e8f0">Usuario:</span></div>
          <div class="hq-field"><input type="text" class="form-control" value="<?= esc($usuario ?? 'DARONDON') ?>" readonly></div>
        </div>

        <div class="hq-inline-hint">Va a realizar cambios en <strong class="hq-count">50</strong> códigos, ¿estás seguro?</div>

        <div class="hq-actions">
          <button type="button" class="btn btn-default" data-dismiss="modal">Cancelar</button>
          <button type="button" class="btn">Aceptar</button>
        </div>
      </form>
    </div>
  </div></div>
</div>

<!-- ===== MODAL: Cargar costos ===== -->
<div id="mdlCostos" class="modal fade hq-modal" tabindex="-1" role="dialog" aria-hidden="true" data-count="50">
  <div class="modal-dialog"><div class="modal-content">
    <div class="modal-header">
      <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
      <h4 class="modal-title">HQ — Cargar costos</h4>
    </div>
    <div class="modal-body">
      <form>
        <div class="hq-row">
          <div class="hq-chip purple">Hasta cuando:</div>
          <div class="hq-field">
            <div class="hq-date-wrap">
              <input type="date" class="form-control">
              <span class="glyphicon glyphicon-calendar"></span>
            </div>
          </div>
          <div style="width:140px; padding-left:14px">
            <label style="font-weight:600; margin:0 0 6px; display:block">Promoción</label>
            <div class="checkbox" style="margin:0"><label style="font-weight:600"><input type="checkbox"></label></div>
          </div>
        </div>

        <div class="hq-row">
          <div class="hq-chip yellow">Motivo de cambio:</div>
          <div class="hq-field"><input type="text" class="form-control"></div>
        </div>

        <div class="hq-row">
          <div class="hq-chip peach"><span style="display:inline-block;background:#fff;padding:2px 8px;border-radius:6px;border:1px solid #e2e8f0">Usuario:</span></div>
          <div class="hq-field"><input type="text" class="form-control" value="<?= esc($usuario ?? 'DARONDON') ?>" readonly></div>
        </div>

        <div class="hq-inline-hint">Va a realizar cambios en <strong class="hq-count">50</strong> códigos, ¿estás seguro?</div>

        <div class="hq-actions">
          <button type="button" class="btn btn-default" data-dismiss="modal">Cancelar</button>
          <button type="button" class="btn">Aceptar</button>
        </div>
      </form>
    </div>
  </div></div>
</div>

    <div class="card-body p-0" style="position:relative">
      <!-- Overlay de carga -->
      <div id="grid-loader" style="position:absolute;inset:0;background:rgba(255,255,255,.88);display:flex;flex-direction:column;justify-content:center;align-items:center;z-index:5">
        <div class="spinner-border text-primary" role="status" style="width:2.0rem;height:2.0rem"></div>
        <div class="mt-3" style="width:min(520px,90%);">
          <div class="progress" style="height:8px">
            <div id="grid-progress" class="progress-bar" role="progressbar" style="width: 0%"></div>
          </div>
        </div>
        <div id="grid-progress-text" class="text-muted mt-2" style="font-size:.9rem">Preparando…</div>
      </div>

      <!-- Contenedor Handsontable -->
      <div id="matriz-grid" class="excel2007" style="height: calc(100vh - 260px); background:#fff"></div>
    </div>
  </div>

  <small class="text-muted d-block mt-2">
    <span style="display:inline-block;width:12px;height:12px;background:#fff;border:1px solid #ddd;margin-right:4px;vertical-align:middle"></span> Editable &nbsp;&nbsp;
    <span style="display:inline-block;width:12px;height:12px;background:#f9fbff;border:1px solid #e0e7ff;margin-right:4px;vertical-align:middle"></span> Fórmula/solo lectura
  </small>
</div>

<!-- ====== Assets ====== -->
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/handsontable@14.4.0/dist/handsontable.full.min.css" />
<script src="https://cdn.jsdelivr.net/npm/handsontable@14.4.0/dist/handsontable.full.min.js"></script>

<script src="https://cdn.jsdelivr.net/npm/numbro@2.3.6/dist/numbro.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/numbro@2.3.6/dist/languages.min.js"></script>
<script>
  (function(){
    var nb = window.numbro && (window.numbro.default || window.numbro);
    if (nb) { try { nb.setLanguage('es-ES'); } catch(e) {} }
  })();
</script>

<style>
  /* ===== Excel moderno ultra-compacto (base) ===== */
  :root{
    --xl-blue:#1976d2; 
    --xl-blue-50:#eef5ff;
    --xl-blue-100:#e2ecfb;
    --xl-border:#c7dbf7; 
    --xl-grid:#e6ebf5;
    --xl-head-text:#0d47a1;
  }

  /* Cabeceras compactas */
  #matriz-grid .handsontable th,
  #matriz-grid .handsontable .ht_clone_top th,
  #matriz-grid .handsontable .ht_clone_left th {
    background: linear-gradient(180deg, var(--xl-blue-50), var(--xl-blue-100));
    color: var(--xl-head-text);
    font-weight: 600;
    border-color: var(--xl-border) !important;
    height: 16px;
    line-height: 16px;
    padding: 0 0px;
    white-space: nowrap;
    font-size: 10.5px;
  }

  /* Bordes finos */
  #matriz-grid .handsontable table,
  #matriz-grid .handsontable td,
  #matriz-grid .handsontable th {
    border-color: var(--xl-grid) !important;
  }

  /* Celdas súper compactas */
  #matriz-grid .handsontable td {
    padding: 0 0px;
    line-height: 1.05;
    height: 16px;
    font-size: 10.5px;
    font-family: Calibri, Roboto, system-ui, -apple-system, "Segoe UI", Arial, sans-serif;
  }
  #matriz-grid .handsontable td.htRight { padding-right: 3px !important; }

  .htRight { text-align: right !important; }

  .cell-manual  { background:#fff !important; }
  .cell-formula { background:#f9fbff !important; color:#222; }
  .cell-manual:focus-within { outline: 2px solid rgba(25,118,210,.25); outline-offset:-2px; }

  .xl-col-badge {
    font-size: 10px; font-weight: 600; color: #0d47a1;
    background:#e9f2ff; border:1px solid #cfe1ff;
    padding: 0 .25rem; border-radius: .25rem; margin-left: .25rem;
  }

  .handsontable .htMenu { font-size: 12px; }
  .handsontable .colHeader .htDropdownMenuButton {
    opacity: 0; transition: opacity .15s ease; transform: scale(0.85); margin-left: 2px;
  }
  .handsontable .colHeader:hover .htDropdownMenuButton { opacity: 1; }

  #matriz-grid .handsontable .ht_master .wtHolder { scrollbar-gutter: stable; }
  #matriz-grid .handsontable .colHeader { font-size: 10.5px; }
  #matriz-grid .handsontable .rowHeader { padding: 0 4px; font-size: 10.5px; }
  #matriz-grid .handsontable .wtBorder.current { border-width: 1px !important; }

  /* ===== Tema Excel 2007 (se aplica con class="excel2007") ===== */
  #matriz-grid.excel2007 {
    --xl07-head-top:   #d4e4ef;
    --xl07-head-mid:   #d4e4ef;
    --xl07-head-bot:   #b7c3cc;
    --xl07-head-text:  #0d2a44;
    --xl07-border-h:   #9EB6CE;
    --xl07-border-c:   #D0D7E5;
    --xl07-row-hover:  #f3f7fb;
    --xl07-edit-outline: rgba(25,118,210,.22);
    --xl07-font: Calibri, Roboto, system-ui, -apple-system, "Segoe UI", Arial, sans-serif;
  }

  #matriz-grid.excel2007 .handsontable td,
  #matriz-grid.excel2007 .handsontable th {
    font-family: var(--xl07-font);
    font-size: 10.5px;
    line-height: 16px;
    height: 16px;
    padding: 0 4px;
    border-color: var(--xl07-border-c) !important;
    white-space: nowrap;
  }

  #matriz-grid.excel2007 .handsontable th,
  #matriz-grid.excel2007 .handsontable .ht_clone_top th,
  #matriz-grid.excel2007 .handsontable .ht_clone_left th {
    color: var(--xl07-head-text);
    font-weight: 600;
    background: linear-gradient(to bottom,
      var(--xl07-head-top) 0%,
      var(--xl07-head-mid) 35%,
      var(--xl07-head-bot) 100%) !important;
    border: 1px solid var(--xl07-border-h) !important;
    height: 17px; line-height: 17px; padding: 0 4px;
  }

  #matriz-grid.excel2007 .handsontable tbody tr:hover td { background-color: var(--xl07-row-hover) !important; }

  #matriz-grid.excel2007 .cell-manual:focus-within {
    outline: 2px solid var(--xl07-edit-outline); outline-offset: -2px;
  }

  #matriz-grid.excel2007 .htRight { text-align:right !important; }
</style>

<script>
(function () {
  // --- ENDPOINTS & CSRF ---
  const URL_DATA = '<?= site_url("matrices2/{$mat['mat_id']}/data") ?>';
  const URL_SAVE = '<?= site_url("matrices2/{$mat['mat_id']}/save-manual") ?>';
  const URL_EVAL = '<?= site_url("matrices2/{$mat['mat_id']}/eval") ?>';
  const CSRF_KEY = '<?= csrf_token() ?>';
  const CSRF_VAL = '<?= csrf_hash() ?>';

  // --- COLUMNAS DINÁMICAS (PHP -> JS) ---
  let dynCols = <?= json_encode($dynCols ?? [], JSON_UNESCAPED_UNICODE) ?>;

  // Limpieza defensiva de duplicados
  (function(){
    const baseProps = new Set(['pro_codigo','pro_descripcion','pro_familia','pro_categorizacion']);
    const unique = [];
    const seen = new Set();
    for (const c of (dynCols || [])) {
      if (!c || !c.name) continue;
      const nm = String(c.name);
      if (baseProps.has(nm)) continue;
      if (seen.has(nm)) continue;
      seen.add(nm);
      unique.push(c);
    }
    dynCols = unique;
  })();

  // Campos editables (no-fórmula) -> igualdad estricta
  const EDITABLE_FIELDS = new Set(
    (dynCols || [])
      .filter(c => c.editable === true && String(c.origen).toLowerCase() !== 'formula')
      .map(c => String(c.name))
  );

  // UI
  const $loader   = document.getElementById('grid-loader');
  const $progBar  = document.getElementById('grid-progress');
  const $progTxt  = document.getElementById('grid-progress-text');
  const $miniStat = document.getElementById('mini-stats');
  const $export   = document.getElementById('btn-export');
  const $mount    = document.getElementById('matriz-grid');

  // Formateadores
  const nf4 = new Intl.NumberFormat('es-EC', { minimumFractionDigits: 4, maximumFractionDigits: 4 });
  function fmtNumber4(v){ if(v==null||v==='') return ''; const n=Number(v); return isFinite(n)?nf4.format(n):''; }
  function fmtPercent4(v){ if(v==null||v==='') return ''; const n=Number(v); return isFinite(n)?nf4.format(n*100)+'%':''; }

  // Renderers
  function rightNumberRenderer(instance, td, row, col, prop, value) {
    Handsontable.renderers.TextRenderer.apply(this, arguments);
    td.classList.add('htRight'); td.textContent = fmtNumber4(value);
  }
  function percentRenderer(instance, td, row, col, prop, value) {
    Handsontable.renderers.TextRenderer.apply(this, arguments);
    td.classList.add('htRight'); td.textContent = fmtPercent4(value);
  }

  // Columnas base fijas
  const BASE_COLS = [
    { prop: 'pro_codigo',         title: 'Código',         width: 80 },
    { prop: 'pro_descripcion',    title: 'Descripción',    width: 100 },
    { prop: 'pro_familia',        title: 'Familia',        width: 80 },
    { prop: 'pro_categorizacion', title: 'Categorización', width: 80 },
  ];

  // Construcción columnas
  function buildColumns() {
    const cols = [];
       const headers = [];

    // pro_id oculto
    cols.push({ data: 'pro_id', readOnly: true, width: 1 });
    headers.push('');

    BASE_COLS.forEach(c => { cols.push({ data:c.prop, readOnly:true, width:c.width }); headers.push(c.title); });

    (dynCols || []).forEach(c => {
      const tipo = String(c.tipo || '').toLowerCase();
      const isEditable = EDITABLE_FIELDS.has(String(c.name));
      const isFormula  = String(c.origen || '').toLowerCase() === 'formula';

      let renderer = 'text', className = '';
      if (tipo === 'percent') { renderer = percentRenderer; className = 'htRight'; }
      else if (/(number|money)/i.test(tipo)) { renderer = rightNumberRenderer; className = 'htRight'; }

      cols.push({
        data: c.name,
        readOnly: isFormula ? true : !isEditable,
        editor: (isFormula ? false : (isEditable ? 'text' : false)),
        renderer, className: (isFormula ? 'cell-formula ' : 'cell-manual ') + className,
        width: 100,
      });

      let h = c.title || c.name;
      if (isFormula) h += ' <span class="xl-col-badge">fx</span>';
      else if (isEditable) h += ' <span class="xl-col-badge">ed</span>';
      headers.push(h);
    });

    return { columns: cols, headers };
  }

  // Estado
  let hot = null;
  let colIndexByProp = Object.create(null);

  function initGrid(initialRows) {
    const cfg = buildColumns();

    colIndexByProp = {};
    cfg.columns.forEach((c, i) => { if (c.data) colIndexByProp[c.data] = i; });

    hot = new Handsontable($mount, {
      data: initialRows,
      columns: cfg.columns,
      colHeaders: cfg.headers,
      rowHeaders: true,
      licenseKey: 'non-commercial-and-evaluation',
      height: 'calc(100vh - 260px)',
      stretchH: 'all',
      manualColumnResize: true,
      manualRowResize: false,
      columnSorting: true,
      autoWrapRow: false,
      autoWrapCol: false,
      wordWrap: false,
      className: 'htCenter',
      afterChange: onCellChange,

      /* ===== (UI) Plugins activados ===== */
      filters: true,
      dropdownMenu: {
        items: {
          'filter_by_condition': {},
          'filter_by_value': {},
          'filter_action_bar': {}
        }
      },
      manualColumnMove: true,
      hiddenColumns: {
        columns: [],
        indicators: true
      },
      contextMenu: {
        items: {
          'row_above': { name: 'Insertar fila arriba' },
          'row_below': { name: 'Insertar fila abajo' },
          '---------': '---------',
          'undo': { name: 'Deshacer' },
          'redo': { name: 'Rehacer' },
          '---------': '---------',

          // Ocultar/mostrar columna (plugin)
          ocultar_columna: {
            name: 'Ocultar columna',
            callback: function(_, selection) {
              if (!selection || !selection.length) return;
              const sel = selection[0];
              const colIndex = sel.start.col;
              const plugin = hot.getPlugin('hiddenColumns');
              plugin.hideColumn(colIndex);
              hot.render();
            }
          },
          mostrar_columna: {
            name: 'Mostrar columna',
            callback: function() {
              const plugin = hot.getPlugin('hiddenColumns');
              const count = hot.countCols();
              for (let c = 0; c < count; c++) {
                if (plugin.isHidden(c)) { plugin.showColumn(c); break; }
              }
              hot.render();
            }
          },
          mostrar_todas: {
            name: 'Mostrar todas',
            callback: function() {
              const plugin = hot.getPlugin('hiddenColumns');
              const count = hot.countCols();
              const toShow = [];
              for (let c = 0; c < count; c++) if (plugin.isHidden(c)) toShow.push(c);
              if (toShow.length) plugin.showColumns(toShow);
              hot.render();
            }
          }
        }
      }
    });

    // compact extra
    const style = document.createElement('style');
    style.textContent = `
      .handsontable .ht_master .wtHolder { scrollbar-gutter: stable; }
      .handsontable col.header, .handsontable th { white-space: nowrap; }
    `;
    document.head.appendChild(style);

    // (Sin coloreado por columna)
    hot.updateSettings({
      afterGetColHeader(col, TH) { /* no-op */ },
      cells() { return {}; }
    });

    hot.addHook('afterColumnMove', function() { hot.render(); });
  }

  // Normaliza valor
  function normalizeForSave(prop, value) {
    const colDef = (dynCols || []).find(c => c.name === prop);
    if (!colDef) return value;
    const tipo = String(colDef.tipo || '').toLowerCase();

    if (tipo === 'percent') {
      if (value == null || value === '') return null;
      const raw = String(value).replace('%','').replace(/\s+/g,'').replace(',','.');
      const num = Number(raw); return isFinite(num) ? (num/100.0) : null;
    }
    if (tipo === 'number' || tipo === 'money') {
      if (value == null || value === '') return null;
      const raw = String(value).replace(',','.'); const num = Number(raw);
      return isFinite(num) ? num : null;
    }
    return value;
  }

  // Guardar + Recalc
  async function onCellChange(changes, source) {
    if (!changes || source === 'loadData' || source === 'recalc') return;
    const toSave = [];
    for (const [row, prop, oldVal, newVal] of changes) {
      if (prop && EDITABLE_FIELDS.has(String(prop)) && oldVal !== newVal) {
        const rowData = hot.getSourceDataAtRow(row);
        if (!rowData || !rowData.pro_id) continue;
        const norm = normalizeForSave(prop, newVal);
        toSave.push({ pro_id: rowData.pro_id, name: prop, value: norm });
      }
    }
    if (!toSave.length) return;

    try {
      for (const item of toSave) {
        await saveManual(item.pro_id, item.name, item.value);
        const recalc = await evalRow(item.pro_id, { [item.name]: item.value });
        if (recalc && recalc.row) updateRowByProId(item.pro_id, recalc.row);
      }
    } catch (err) {
      console.error('save/eval error', err);
      alert('Ocurrió un error al guardar o recalcular:\n' + (err?.message || err));
    }
  }

  async function saveManual(proId, name, value) {
    const body = new URLSearchParams();
    body.append('pro_id', proId);
    body.append('name', name);
    body.append('value', value === null ? '' : String(value));
    body.append(CSRF_KEY, CSRF_VAL);
    const r = await fetch(URL_SAVE, { method: 'POST', body });
    if (!r.ok) throw new Error('save HTTP ' + r.status);
    const j = await r.json();
    if (!j.ok) throw new Error('save not ok');
    return true;
  }

  async function evalRow(proId, overridesObj) {
    const body = new URLSearchParams();
    body.append('pro_id', proId);
    body.append('overrides', JSON.stringify(overridesObj || {}));
    body.append(CSRF_KEY, CSRF_VAL);
    const r = await fetch(URL_EVAL, { method: 'POST', body });
    if (!r.ok) throw new Error('eval HTTP ' + r.status);
    return r.json();
  }

  function updateRowByProId(proId, newRow) {
    const data = hot.getSourceData();
    const idx  = data.findIndex(r => String(r.pro_id) === String(proId));
    if (idx < 0) return;
    Object.keys(newRow).forEach(prop => {
      if (prop in colIndexByProp) {
        const col = colIndexByProp[prop];
        hot.setDataAtCell(idx, col, newRow[prop], 'recalc');
      }
    });
  }

  /* ===== Exportación corregida (locale es-EC, % y números sin comillas) ===== */
  function exportCSV_excelLocale() {
    if (!hot) return;

    const hidden = hot.getPlugin('hiddenColumns');
    const colCount = hot.countCols();
    const rowCount = hot.countRows();

    // Tipo por prop (para formatear)
    const tipoByProp = {};
    (dynCols || []).forEach(c => { if (c && c.name) tipoByProp[c.name] = String(c.tipo || 'text').toLowerCase(); });
    // Base:
    tipoByProp['pro_codigo']         = 'text';
    tipoByProp['pro_descripcion']    = 'text';
    tipoByProp['pro_familia']        = 'text';
    tipoByProp['pro_categorizacion'] = 'text';

    // Columnas visibles en orden visual
    const visibleCols = [];
    for (let c = 0; c < colCount; c++) {
      if (hidden && hidden.isHidden && hidden.isHidden(c)) continue;
      const prop = (hot.getSettings().columns?.[c]?.data) || null;
      const headerRaw = hot.getColHeader(c);
      const headerTxt = (typeof headerRaw === 'string'
        ? headerRaw.replace(/<[^>]*>/g, '').trim()
        : String(headerRaw ?? '').trim());
      if (prop === 'pro_id') continue; // no exportar id
      if (!headerTxt) continue;
      visibleCols.push({ idx: c, prop, header: headerTxt });
    }

    // Encabezados
    const out = [];
    out.push(visibleCols.map(col => col.header).join(';'));

    // Formateadores (coma decimal)
    const nf4 = new Intl.NumberFormat('es-EC', { minimumFractionDigits: 4, maximumFractionDigits: 4 });
    const fmtNum = v => {
      if (v==null || v==='') return '';
      const n = Number(String(v).replace(/\s+/g,'').replace(',', '.'));
      return isFinite(n) ? nf4.format(n) : '';
    };
    const fmtPct = v => {
      if (v==null || v==='') return '';
      const n = Number(String(v).replace(/\s+/g,'').replace(',', '.'));
      return isFinite(n) ? nf4.format(n*100) + '%' : '';
    };

    // Filas
    for (let r = 0; r < rowCount; r++) {
      const src = hot.getSourceDataAtRow(r) || {};
      const row = visibleCols.map(col => {
        const tipo = (tipoByProp[col.prop] || 'text');
        let v = (col.prop ? src[col.prop] : hot.getDataAtCell(r, col.idx));

        if (tipo === 'percent') {
          return fmtPct(v); // 12,3456%
        } else if (tipo === 'number' || tipo === 'money') {
          return fmtNum(v); // 1.234,0000
        } else {
          v = (v==null ? '' : String(v));
          v = v.replace(/\r?\n/g, ' ').replace(/;/g, ',').replace(/"/g, '""');
          return `"${v}"`;
        }
      });
      out.push(row.join(';'));
    }

    // BOM UTF-8 para Excel
    const csv = '\uFEFF' + out.join('\n');
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8' });
    const a = document.createElement('a');
    a.href = URL.createObjectURL(blob);
    a.download = 'matriz_<?= $matId ?>.csv';
    document.body.appendChild(a); a.click(); a.remove();
  }
  if ($export) $export.addEventListener('click', exportCSV_excelLocale);

  // -------- CARGA TOTAL (chunks adaptativos) --------
  let CHUNK_SIZE = 5000;
  const MIN_CHUNK = 500;
  const MAX_RETRY = 3;

  async function fetchChunk(start, length, attempt = 1) {
    const body = new URLSearchParams();
    body.append('draw','1'); body.append('start', String(start)); body.append('length', String(length));
    body.append('search[value]',''); body.append(CSRF_KEY, CSRF_VAL);
    try {
      const r = await fetch(URL_DATA, { method:'POST', body });
      if (!r.ok) throw new Error('HTTP ' + r.status);
      return await r.json();
    } catch (err) {
      if (attempt < MAX_RETRY && length > MIN_CHUNK) {
        const nextLen = Math.max(MIN_CHUNK, Math.floor(length*0.66));
        await new Promise(res => setTimeout(res, 500*attempt));
        return fetchChunk(start, nextLen, attempt+1);
      }
      throw err;
    }
  }

  async function loadAll() {
    if ($loader) $loader.style.display = 'flex';
    if ($progTxt) $progTxt.textContent = 'Cargando…';
    if ($progBar) $progBar.style.width = '0%';

    let start = 0, total = null, loaded = 0, rows = [];

    while (true) {
      const resp = await fetchChunk(start, CHUNK_SIZE);
      const data = resp.data || [];
      if (total === null) total = resp.recordsFiltered || resp.recordsTotal || data.length;

      if (data.length) {
        rows = rows.concat(data);
        loaded += data.length;

        if (!hot) initGrid(rows);
        else hot.updateData(rows);

        const pct = Math.min(100, Math.round((loaded / Math.max(1, total)) * 100));
        if ($progBar) $progBar.style.width = pct + '%';
        if ($progTxt) $progTxt.textContent = `Cargando ${loaded.toLocaleString()} / ${total.toLocaleString()}…`;
        if ($miniStat) $miniStat.textContent = `${loaded.toLocaleString()} filas cargadas`;
      }

      if (data.length === 0) break;
      start += data.length;
      if (loaded >= (total ?? 0)) break;
    }

    if ($loader) $loader.style.display = 'none';
  }

  loadAll();
})();
</script>
<script>
  // Define el conteo dinámico desde JS cuando abras el modal
  // Cambia esta variable cuando tengas el número real
  window.MATRIZ_CHANGE_COUNT = 50;

  function applyCount(modal){
    var count = (typeof window.MATRIZ_CHANGE_COUNT === 'number') ? window.MATRIZ_CHANGE_COUNT : 50;
    var attr = modal.getAttribute('data-count');
    if (attr && !isNaN(+attr)) count = +attr;
    var nodes = modal.querySelectorAll('.hq-count');
    for (var i=0; i<nodes.length; i++){ nodes[i].textContent = count; }
  }

  $('#mdlDist, #mdlPDV, #mdlCostos').on('show.bs.modal', function(){
    applyCount(this);
  });
</script>
<script>
(function(){
  const MAT_ID  = <?= (int)$mat['mat_id'] ?>;
  const CSRF_KEY= '<?= csrf_token() ?>';
  const CSRF_VAL= '<?= csrf_hash() ?>';

  async function snapshot(tipo, {hasta, promo, motivo, usuario}){
    const body = new URLSearchParams();
    body.append('tipo', tipo);
    body.append('hasta', hasta || '');
    body.append('promo', promo ? '1' : '0');
    body.append('motivo', motivo || '');
    body.append('usuario', usuario || '<?= esc($usuario ?? 'DARONDON') ?>');
    body.append(CSRF_KEY, CSRF_VAL);

    const url = '<?= site_url("matrices2/{$mat['mat_id']}/versions/create") ?>';
    const r = await fetch(url, { method:'POST', body });
    if (!r.ok) throw new Error('Snapshot HTTP '+r.status);
    const j = await r.json();
    if (!j.ok) throw new Error('Snapshot no OK');
    return j; // {ok:true, mver_id, rows}
  }

  function bindModal(modalId, tipo){
    const $m = document.getElementById(modalId);
    if (!$m) return;
    $m.addEventListener('click', async function(e){
      const target = e.target.closest('button.btn'); // tu botón "Aceptar" tiene class="btn"
      if (!target) return;
      if (target.textContent.trim().toLowerCase().indexOf('aceptar') === -1) return;

      // lee campos del modal
      const hasta = $m.querySelector('input[type="date"]')?.value || '';
      const promo = !!$m.querySelector('input[type="checkbox"]')?.checked;
      const motivo= ($m.querySelector('input[type="text"]')?.value || '').trim();
      const usuario = $m.querySelector('input[readonly]')?.value || 'SYSTEM';

      target.disabled = true; target.textContent = 'Guardando versión…';
      try {
        const res = await snapshot(tipo, {hasta, promo, motivo, usuario});
        // feedback
        alert('Versión guardada (#'+res.mver_id+') con '+res.rows.toLocaleString()+' filas.');
        // aquí podrías llamar a tu endpoint real de “carga” si ya lo tienes implementado
        // await fetch('...aplicar-cambios...', { method:'POST', body: ... });
        $($m).modal('hide');
      } catch(err){
        console.error(err);
        alert('Error al versionar: '+(err?.message||err));
      } finally {
        target.disabled = false; target.textContent = 'Aceptar';
      }
    });
  }

  bindModal('mdlDist',   'DIST');
  bindModal('mdlPDV',    'PDV');
  bindModal('mdlCostos', 'COSTOS');

})();
</script>
