<?php
namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\CampoGlobalModel;
use App\Libraries\FieldResolver;

class CamposController extends BaseController
{
    protected $db;
    protected $campos;

    public function __construct()
    {
        $this->db     = \Config\Database::connect('postgres');
        $this->campos = new CampoGlobalModel();
    }

    // UI listado
    public function index()
    {
        echo view('layouts/header');
        echo view('layouts/aside');
        echo view('campos/index');
        echo view('layouts/footer');
    }

    // API: listar
    public function list()
    {
        $rows = $this->campos->orderBy('cam_nombre','asc')->findAll();
        return $this->response->setJSON(['ok'=>true,'data'=>$rows]);
    }

    // API: crear
    public function save()
    {
        $p = $this->request->getPost();
        $data = [
            'cam_nombre'  => trim($p['cam_nombre'] ?? ''),
            'cam_titulo'  => trim($p['cam_titulo'] ?? ''),
            'cam_origen'  => trim($p['cam_origen'] ?? 'lectura'),
            'cam_tipo'    => trim($p['cam_tipo']   ?? 'text'),
            'cam_formato' => $p['cam_formato'] ?? null,
            'cam_color'   => $p['cam_color']   ?? null,
            'cam_activo'  => (int)($p['cam_activo'] ?? 1),
        ];

        if ($data['cam_nombre']==='' || $data['cam_titulo']==='') {
            return $this->response->setStatusCode(422)->setJSON(['ok'=>false,'msg'=>'Nombre y Título requeridos']);
        }

        $depends = $p['cam_depends'] ?? [];
        if (is_string($depends)) $depends = json_decode($depends,true) ?: [];
        $data['cam_depends'] = json_encode(array_values(array_filter($depends)), JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);

        $src = trim((string)($p['cam_source'] ?? ''));
        $data['cam_source'] = $src !== '' ? $src : null;

        $formula = trim((string)($p['cam_formula'] ?? ''));
        $data['cam_formula'] = $formula !== '' ? $formula : null;

        try{
            $this->campos->insert($data);
            return $this->response->setJSON(['ok'=>true,'id'=>$this->campos->getInsertID()]);
        } catch(\Throwable $e){
            return $this->response->setStatusCode(500)->setJSON(['ok'=>false,'msg'=>$e->getMessage()]);
        }
    }

    // API: actualizar
    public function update($camId)
    {
        $p = $this->request->getPost();
        $row = $this->campos->find((int)$camId);
        if (!$row) return $this->response->setStatusCode(404)->setJSON(['ok'=>false,'msg'=>'Campo no encontrado']);

        $data = [
            'cam_titulo'  => trim($p['cam_titulo'] ?? $row['cam_titulo']),
            'cam_origen'  => trim($p['cam_origen'] ?? $row['cam_origen']),
            'cam_tipo'    => trim($p['cam_tipo']   ?? $row['cam_tipo']),
            'cam_formato' => $p['cam_formato'] ?? $row['cam_formato'],
            'cam_color'   => $p['cam_color']   ?? $row['cam_color'],
            'cam_activo'  => (int)($p['cam_activo'] ?? $row['cam_activo']),
        ];

        $depends = $p['cam_depends'] ?? null;
        if ($depends !== null) {
            if (is_string($depends)) $depends = json_decode($depends,true) ?: [];
            $data['cam_depends'] = json_encode(array_values(array_filter($depends)), JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
        }

        if (array_key_exists('cam_source',$p)) {
            $src = trim((string)($p['cam_source'] ?? ''));
            $data['cam_source'] = $src !== '' ? $src : null;
        }
        if (array_key_exists('cam_formula',$p)) {
            $formula = trim((string)($p['cam_formula'] ?? ''));
            $data['cam_formula'] = $formula !== '' ? $formula : null;
        }

        try{
            $this->campos->update((int)$camId, $data);
            return $this->response->setJSON(['ok'=>true]);
        } catch(\Throwable $e){
            return $this->response->setStatusCode(500)->setJSON(['ok'=>false,'msg'=>$e->getMessage()]);
        }
    }

    // API: eliminar
    public function delete($camId)
    {
        try{
            $this->campos->delete((int)$camId);
            return $this->response->setJSON(['ok'=>true]);
        } catch(\Throwable $e){
            return $this->response->setStatusCode(500)->setJSON(['ok'=>false,'msg'=>$e->getMessage()]);
        }
    }

    // API: probar (con pro_id)
    public function test($camId)
    {
        $resolver = new FieldResolver();
        $cam = $this->campos->find((int)$camId);
        if (!$cam) return $this->response->setStatusCode(404)->setJSON(['ok'=>false,'msg'=>'Campo no encontrado']);

        $proId = (int)($this->request->getPost('pro_id') ?? 0);
        if (!$proId) return $this->response->setStatusCode(422)->setJSON(['ok'=>false,'msg'=>'pro_id requerido']);

        $prod = $this->db->table('public.tbl_producto')->where('pro_id',$proId)->get()->getRowArray();
        if (!$prod) return $this->response->setStatusCode(404)->setJSON(['ok'=>false,'msg'=>'Producto no encontrado']);

        $ctx = [];
        // precargar lecturas directas que podrían referenciarse en fórmulas
        // (opcional: podrías resolver dependencias verdaderas; aquí hacemos simple)
        $val = $resolver->resolveField($cam, $prod, null, $ctx);
        return $this->response->setJSON(['ok'=>true,'value'=>$val]);
    }
}
