<?php
namespace App\Controllers;

use App\Models\ProductoModel;
use App\Models\ProductoStockModel;
use App\Models\SyncLogProductoModel;

class Sincronizacion extends BaseController
{
    private $baseUri;
    private $authRaw;
    private $pageSize   = 500; // tamaño base recomendado
    private $maxRetries = 2;   // reintentos por página
    private $timeout    = 60;  // segundos

    public function __construct()
    {
        $this->baseUri = rtrim(getenv('precios.base_uri') ?: 'https://int.api.megaprofer.com/', '/') . '/';
        $this->authRaw = getenv('precios.auth_raw') ?: 'rapidito-int;N4RaVsn|&BaXLTIk]1';
        $this->timeout = (int) (getenv('precios.timeout') ?: $this->timeout);
    }

    // GET /sincronizar/productos
    public function productos()
    {
        @set_time_limit(0);
        @ini_set('memory_limit', '1024M');

        $mLog  = new \App\Models\SyncLogProductoModel();
        $mSync = new \App\Models\PreciosSyncModel();
        $mProd = new \App\Models\ProductoModel();

        // Conexión separada para LOG (no se contamina)
        $dbLog = \Config\Database::connect('postgres', true);

        $logId = $mLog->startRun($this->pageSize);

        $totPages=0; $totElems=0; $ins=0; $upd=0; $inact=0; $errs=0;
        $runMark = date('Y-m-d H:i:s');
        $heartbeatEvery = 75;

        try {
            $first    = $this->fetchPageWithRetry(0);
            $totPages = (int)($first['totalPages'] ?? 0);
            $totElems = (int)($first['totalElements'] ?? 0);

            for ($p=0; $p<$totPages; $p++) {
                try {
                    $page = $p===0 ? $first : $this->fetchPageWithRetry($p);
                    $data = $page['data'] ?? [];
                    if (empty($data)) continue;

                    [$rows, $stocks] = $this->mapPage($data, $runMark);
                    $res = $mSync->syncPage($rows, $stocks);

                    if (empty($res['ok'])) {
                        $errs++;
                        log_message('error', 'Sync pág '.$p.' error BD: '.($res['error']['message'] ?? ''));
                        continue;
                    }
                    $ins += $res['inserted'] ?? 0;
                    $upd += $res['updated'] ?? 0;

                    // Heartbeat
                    if (($p % $heartbeatEvery) === 0) {
                        $dbLog->table('public.tbl_sync_log_producto')->where('slog_id', $logId)->update([
                            'message' => "heartbeat base='' page=$p/$totPages ins=$ins upd=$upd errs=$errs"
                        ]);
                    }
                } catch (\Throwable $ePage) {
                    $errs++;
                    log_message('error', 'Sync pág '.$p.' excepción: '.$ePage->getMessage());
                    continue;
                }
            }

            if ($errs===0) {
                $inact = $mProd->inactivateNotSeenSince($runMark);
            }

            // LOG (conexión limpia)
            $dbLog->table('public.tbl_sync_log_producto')->where('slog_id', $logId)->update([
                'finished_at'    => date('Y-m-d H:i:s'),
                'total_pages'    => $totPages,
                'total_elements' => $totElems,
                'inserted'       => $ins,
                'updated'        => $upd,
                'inactivated'    => $inact,
                'errors'         => $errs,
                'status'         => $errs>0 ? 'PARTIAL' : 'OK',
                'message'        => $errs>0 ? 'Sync parcial: ver logs.' : 'Sync completa OK',
            ]);

            return $this->response->setJSON([
                'ok'=>true, 'pages'=>$totPages, 'total'=>$totElems,
                'ins'=>$ins, 'upd'=>$upd, 'inact'=>$inact, 'errs'=>$errs
            ]);

        } catch (\Throwable $e) {
            // si algo explota, deja constancia en LOG
            try {
                $dbLog->table('public.tbl_sync_log_producto')->where('slog_id', $logId)->update([
                    'finished_at'=>date('Y-m-d H:i:s'),
                    'status'=>'ERROR',
                    'message'=>$e->getMessage()
                ]);
            } catch (\Throwable $e2) {}
            log_message('error', 'Sync productos ERROR global: '.$e->getMessage());
            return $this->response->setStatusCode(500)->setJSON(['ok'=>false,'error'=>$e->getMessage()]);
        }
    }

    /* ====================== Helpers HTTP (cURL) ====================== */

    private function fetchPageWithRetry(int $page): array
    {
        $attempt = 0;
        while (true) {
            try {
                return $this->fetchPage($page);
            } catch (\Throwable $e) {
                $attempt++;
                if ($attempt > $this->maxRetries) { throw $e; }
                sleep($attempt === 1 ? 2 : 5);
            }
        }
    }

    private function fetchPage(int $page): array
    {
        $endpoint = $this->baseUri . 'pricing/search-products';
        $query = http_build_query([
            'searchValue' => '',
            'page'        => $page,
            'size'        => $this->pageSize,
        ], '', '&', PHP_QUERY_RFC3986);

        $headers = [
            'Accept: application/json',
            'Content-Type: application/json',
            'Authorization: ' . $this->authRaw,
        ];

        $resp = $this->curlGet("$endpoint?$query", $headers, $this->timeout);
        $arr  = json_decode($resp, true);
        if (!is_array($arr)) {
            throw new \RuntimeException('Respuesta inválida del API en page=' . $page);
        }
        return $arr;
    }

    // ---- helpers HTTP para test ----
    private function curlGetWithRetry(string $url, array $headers, int $timeout = 60, int $maxRetries = 2): string
    {
        $reqId = bin2hex(random_bytes(8));
        $headers[] = 'X-Request-Id: '.$reqId;
        $headers[] = 'User-Agent: PreciosSync/1.0';

        $attempt = 0;
        while (true) {
            try {
                return $this->curlGet($url, $headers, $timeout);
            } catch (\RuntimeException $e) {
                $attempt++;
                if ($attempt > $maxRetries) {
                    throw new \RuntimeException('[reqId='.$reqId.'] '.$e->getMessage());
                }
                sleep($attempt === 1 ? 2 : 5);
            }
        }
    }

    private function curlGet(string $url, array $headers, int $timeout = 60): string
    {
        $ch = curl_init();
        if ($ch === false) throw new \RuntimeException('No se pudo inicializar cURL');

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_CONNECTTIMEOUT => $timeout,
            CURLOPT_TIMEOUT        => $timeout,
            CURLOPT_HTTPHEADER     => $headers,
            // Si el INT tiene certificado interno y falla SSL, descomenta temporalmente:
            // CURLOPT_SSL_VERIFYPEER => false,
            // CURLOPT_SSL_VERIFYHOST => false,
        ]);

        $body = curl_exec($ch);
        if ($body === false) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new \RuntimeException('Error cURL: ' . $err);
        }
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($code < 200 || $code >= 300) {
            throw new \RuntimeException("HTTP $code: $body");
        }
        return $body;
    }

    /* ====================== TEST ====================== */
    // GET /sincronizar/test?q=c05050&page=0&size=100
    public function test()
    {
        @set_time_limit(0);

        $q    = (string)($this->request->getGet('q') ?? '');
        $page = (int)($this->request->getGet('page') ?? 0);
        $size = (int)($this->request->getGet('size') ?? 100);

        $baseUri = rtrim(getenv('precios.base_uri') ?: 'https://int.api.megaprofer.com/', '/') . '/';
        $auth    = getenv('precios.auth_raw') ?: 'rapidito-int;N4RaVsn|&BaXLTIk]1';

        $endpoint = $baseUri . 'pricing/search-products';
        $query    = http_build_query(['searchValue'=>$q, 'page'=>$page, 'size'=>$size], '', '&', PHP_QUERY_RFC3986);

        $headers = [
            'Accept: application/json',
            'Content-Type: application/json',
            'Authorization: ' . $auth,
        ];

        try {
            $body = $this->curlGetWithRetry($endpoint . '?' . $query, $headers, 60, 2);
            $arr  = json_decode($body, true);
            return $this->response->setJSON($arr ?? ['raw' => $body]);
        } catch (\Throwable $e) {
            $msg = $e->getMessage();
            $prov = null;
            if (preg_match('/HTTP\s+\d+:\s+(.*)$/s', $msg, $m)) {
                $prov = json_decode($m[1], true);
            }
            return $this->response->setStatusCode(500)->setJSON([
                'ok'        => false,
                'error'     => $msg,
                'provider'  => $prov,
            ]);
        }
    }

    /* ====================== Mapeo a filas BD ====================== */

    private function n6($v) { return $v === null ? null : round((float)$v, 6); }
    private function b($v)  { return is_bool($v) ? $v : (in_array($v, [1,'1','true',true], true)); }

    private function mapPage(array $data, string $runMark): array
    {
        $rows = []; $stocks = [];

        foreach ($data as $it) {
            $refId   = (string)($it['referenceId'] ?? '');
            if ($refId === '') continue;

            $code    = (string)($it['referenceCode'] ?? '');
            $barcode = (string)($it['barcode'] ?? '');
            $name    = (string)($it['name'] ?? '');
            $desc    = (string)($it['description'] ?? '');

            $tax     = $it['taxPresenter'] ?? [];
            $unit    = $it['measureUnitPresenter'] ?? [];
            $subline = $it['productSublinePresenter'] ?? [];
            $familyP = $subline['productFamilyPresenter'] ?? [];
            $mark    = $it['markPresenter'] ?? [];
            $stocksA = $it['stocks'] ?? [];

            $row = [
                'pro_reference_id'        => $refId,
                'pro_codigo'              => $code !== '' ? $code : null,
                'pro_barcode'             => $barcode !== '' ? $barcode : null,
                'pro_descripcion'         => $name ?? '',
                'pro_detalle'             => $desc ?? '',
                'pro_familia'             => (string)($familyP['name'] ?? ($it['family'] ?? '')),
                'pro_sublinea'            => (string)($subline['name'] ?? ''),
                'pro_marca'               => (string)($mark['name'] ?? ''),
                'pro_proveedor_habitual'  => (string)($it['provider'] ?? ''),
                'pro_categorizacion'      => (string)($it['categorization'] ?? ''),
                'pro_unidad_nombre'       => (string)($unit['name'] ?? ''),
                'pro_unidad_simbolo'      => (string)($unit['symbol'] ?? ''),
                'pro_iva_nombre'          => (string)($tax['name'] ?? ''),

                'pro_peso'                => $this->n6($it['weight'] ?? null),
                'pro_iva_valor'           => $this->n6($tax['value'] ?? null),
                'pro_cost'                => $this->n6($it['cost'] ?? null),
                'pro_price_cost'          => $this->n6($it['priceCost'] ?? null),
                'pro_payed_price'         => $this->n6($it['payedPrice'] ?? null),
                'pro_stock'               => $this->n6($it['stock'] ?? null),
                'pro_authorized_discount' => $this->n6($it['authorizedDiscount'] ?? null),

                'pro_activo'              => true,
                'pro_es_pesado'           => $this->b($it['isHeavy'] ?? null),
                'pro_es_especial'         => $this->b($it['isEspecial'] ?? null),
                'pro_need_validate_stock' => $this->b($it['needValidateStock'] ?? null),
                'pro_is_supply'           => $this->b($it['isSupply'] ?? null),

                'pro_location'            => (string)($it['location'] ?? ''),
                'pro_updated_api'         => !empty($it['updatedDate']) ? $it['updatedDate'] : null,
                'pro_seen_at'             => $runMark,
                'pro_inactive_at'         => null,
                'pro_json'                => json_encode($it, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
            ];

            foreach (['pro_descripcion','pro_detalle','pro_familia','pro_sublinea','pro_marca',
                      'pro_proveedor_habitual','pro_categorizacion','pro_unidad_nombre',
                      'pro_unidad_simbolo','pro_iva_nombre','pro_location'] as $k) {
                if ($row[$k] === null) $row[$k] = '';
            }

            $hashBase = [
                $row['pro_codigo'],$row['pro_barcode'],$row['pro_descripcion'],$row['pro_detalle'],
                $row['pro_familia'],$row['pro_sublinea'],$row['pro_marca'],$row['pro_proveedor_habitual'],
                $row['pro_categorizacion'],$row['pro_unidad_nombre'],$row['pro_unidad_simbolo'],$row['pro_iva_nombre'],
                $row['pro_peso'],$row['pro_iva_valor'],$row['pro_cost'],$row['pro_price_cost'],$row['pro_payed_price'],
                $row['pro_stock'],$row['pro_authorized_discount'],$row['pro_es_pesado'],$row['pro_es_especial'],
                $row['pro_need_validate_stock'],$row['pro_is_supply'],$row['pro_location'],$row['pro_updated_api']
            ];
            $row['pro_hash'] = sha1(json_encode($hashBase));

            $rows[] = $row;

            if (is_array($stocksA)) {
                foreach ($stocksA as $st) {
                    $wName = (string)($st['warehouseName'] ?? '');
                    if ($wName === '') continue;
                    $stocks[] = [
                        'pro_reference_id' => $refId,
                        'warehouse_name'   => $wName,
                        'stock'            => $this->n6($st['stock'] ?? 0),
                    ];
                }
            }
        }

        return [$rows, $stocks];
    }

    /* ====================== Prefijos ====================== */

    // GET /sincronizar/productos-prefijo
    public function productos_prefijo()
    {
        @set_time_limit(0);
        @ini_set('memory_limit', '1024M');

        $mLog    = new \App\Models\SyncLogProductoModel();
        $sync    = new \App\Models\PreciosSyncModel();
        $runMark = date('Y-m-d H:i:s');

        $logId = $mLog->insert([
            'started_at' => $runMark,
            'page_size'  => $this->pageSize,
            'status'     => 'RUNNING',
            'message'    => 'prefijos ampliados'
        ], true);

        $ins=0; $upd=0; $errs=0; $pages=0;
        $heartbeatEvery = 75;

        foreach ($this->basePrefixes() as $pf) {
            $r = $this->runAllPagesForPrefix($pf, $runMark, $sync, $logId, $heartbeatEvery);
            $ins+=$r['ins']; $upd+=$r['upd']; $errs+=$r['errs']; $pages+=$r['pages'];

            // si un prefijo trae muchas páginas, subdivide a bigramas
            if ($r['pages'] >= 50) {
                foreach ($this->bigramsFor($pf) as $bi) {
                    $rb = $this->runAllPagesForPrefix($bi, $runMark, $sync, $logId, $heartbeatEvery);
                    $ins+=$rb['ins']; $upd+=$rb['upd']; $errs+=$rb['errs']; $pages+=$rb['pages'];
                }
            }
        }

        // Inactivar SOLO al final del barrido por prefijos y si no hubo errores
        $inact = 0;
        if ($errs === 0) {
            $prod = new \App\Models\ProductoModel();
            $inact = $prod->inactivateNotSeenSince($runMark);
        }

        // cerrar log con conexión limpia
        $dbLog = \Config\Database::connect('postgres', true);
        $dbLog->table('public.tbl_sync_log_producto')->where('slog_id', $logId)->update([
            'finished_at' => date('Y-m-d H:i:s'),
            'total_pages' => $pages,
            'inserted'    => $ins,
            'updated'     => $upd,
            'inactivated' => $inact,
            'errors'      => $errs,
            'status'      => $errs ? 'PARTIAL':'OK',
        ]);

        return $this->response->setJSON([
            'ok' => $errs===0, 'pages'=>$pages, 'ins'=>$ins, 'upd'=>$upd, 'inact'=>$inact, 'errs'=>$errs
        ]);
    }

    private function basePrefixes(): array
    {
        return array_merge(range('0','9'), range('A','Z'), ['Ñ','Á','É','Í','Ó','Ú','Ü','-','_','.','/']);
    }

    private function bigramsFor(string $ch): array
    {
        $out = [];
        foreach (array_merge(range('0','9'), range('A','Z')) as $suf) $out[] = $ch.$suf;
        return $out;
    }

    /** Recorre TODAS las páginas de un prefijo respetando totalPages, con heartbeat */
    private function runAllPagesForPrefix(string $prefix, string $runMark, \App\Models\PreciosSyncModel $sync, int $logId, int $heartbeatEvery = 75): array
    {
        $ins=0;$upd=0;$errs=0;$pages=0; $page=0; $totPages=null;

        while (true) {
            try {
                $pd = $this->fetchPageWithRetryEx($page, $prefix);
                if ($totPages === null) $totPages = (int)($pd['totalPages'] ?? 0);

                $data = $pd['data'] ?? [];
                if (empty($data)) break;

                [$rows,$stocks] = $this->mapPage($data, $runMark);
                $res = $sync->syncPage($rows, $stocks);

                if (empty($res['ok'])) { $errs++; }
                else { $ins += $res['inserted'] ?? 0; $upd += $res['updated'] ?? 0; }

                $pages++; $page++;

                // Heartbeat de avance por prefijo
                if ($pages % $heartbeatEvery === 0) {
                    $dbLog = \Config\Database::connect('postgres', true);
                    $dbLog->table('public.tbl_sync_log_producto')->where('slog_id', $logId)->update([
                        'message' => "heartbeat pf='$prefix' page=$page/".($totPages ?? '?')." ins=$ins upd=$upd errs=$errs"
                    ]);
                }

                if ($totPages && $page >= $totPages) break; // corte por totalPages
            } catch (\Throwable $e) {
                // Ventana base falló: subdividir y recuperar lo posible
                $rec = $this->recoverFailedWindow($prefix, $page, $runMark, $sync);
                $ins+= $rec['ins']; $upd+= $rec['upd']; $errs+= $rec['errs']; $pages+= $rec['pages'];
                $page++; // pasa a la siguiente base
                if ($totPages && $page >= $totPages) break;
            }
        }
        return compact('ins','upd','errs','pages');
    }

    private function fetchPageWithRetryEx(int $page, string $searchValue): array
    {
        $attempt=0;
        while (true) {
            try { return $this->fetchPageEx($page, $searchValue); }
            catch (\Throwable $e) {
                $attempt++; if ($attempt > $this->maxRetries) throw $e;
                sleep($attempt===1?2:5);
            }
        }
    }

    private function fetchPageEx(int $page, string $searchValue): array
    {
        $endpoint = $this->baseUri . 'pricing/search-products';
        $query    = http_build_query([
            'searchValue' => $searchValue,
            'page'        => $page,
            'size'        => $this->pageSize,
        ], '', '&', PHP_QUERY_RFC3986);

        $headers = [
            'Accept: application/json',
            'Content-Type: application/json',
            'Authorization: ' . $this->authRaw,
        ];
        $resp = $this->curlGet($endpoint.'?'.$query, $headers, $this->timeout);
        $arr  = json_decode($resp, true);
        if (!is_array($arr)) throw new \RuntimeException('Respuesta inválida page='.$page);
        return $arr;
    }

    /* ====================== V2: Recuperación Adaptativa ====================== */

    // GET /sincronizar/productos-prefijo-v2
    public function productos_prefijo_v2()
    {
        @set_time_limit(0);
        @ini_set('memory_limit', '1024M');

        $mLog    = new \App\Models\SyncLogProductoModel();
        $sync    = new \App\Models\PreciosSyncModel();
        $prod    = new \App\Models\ProductoModel();
        $runMark = date('Y-m-d H:i:s');

        $logId = $mLog->insert([
            'started_at' => $runMark,
            'page_size'  => $this->pageSize,
            'status'     => 'RUNNING',
            'message'    => 'prefijos v2 (adaptive windows)'
        ], true);

        $ins=0; $upd=0; $errs=0; $pages=0; $winRecovered=0;
        $heartbeatEvery = 75;

        foreach ($this->basePrefixes() as $pf) {
            $page=0; $totPages=null;

            while (true) {
                try {
                    $pd = $this->fetchPageWithRetryEx($page, $pf);
                    if ($totPages === null) $totPages = (int)($pd['totalPages'] ?? 0);

                    $data = $pd['data'] ?? [];
                    if (empty($data)) break;

                    [$rows,$stocks] = $this->mapPage($data, $runMark);
                    $res = $sync->syncPage($rows, $stocks);
                    if (!empty($res['ok'])) { $ins += $res['inserted']??0; $upd += $res['updated']??0; }
                    else { $errs++; }

                    $pages++; $page++;

                    // Heartbeat
                    if ($pages % $heartbeatEvery === 0) {
                        $dbLog = \Config\Database::connect('postgres', true);
                        $dbLog->table('public.tbl_sync_log_producto')->where('slog_id', $logId)->update([
                            'message' => "heartbeat v2 pf='$pf' page=$page/".($totPages ?? '?')." ins=$ins upd=$upd errs=$errs windows=$winRecovered"
                        ]);
                    }

                    if ($totPages && $page >= $totPages) break; // corte por totalPages
                } catch (\Throwable $e) {
                    // Ventana base falló: subdividir y recuperar lo posible
                    $rec = $this->recoverFailedWindow($pf, $page, $runMark, $sync);
                    $ins+= $rec['ins']; $upd+= $rec['upd']; $errs+= $rec['errs']; $pages+= $rec['pages'];
                    $winRecovered++;
                    $page++; // siguiente base
                    if ($totPages && $page >= $totPages) break;
                }
            }
        }

        // Inactivar al final si no hubo errores duros
        $inact = 0;
        if ($errs === 0) {
            $inact = $prod->inactivateNotSeenSince($runMark);
        }

        // cerrar log
        $dbLog = \Config\Database::connect('postgres', true);
        $dbLog->table('public.tbl_sync_log_producto')->where('slog_id', $logId)->update([
            'finished_at' => date('Y-m-d H:i:s'),
            'total_pages' => $pages,
            'inserted'    => $ins,
            'updated'     => $upd,
            'inactivated' => $inact,
            'errors'      => $errs,
            'status'      => $errs ? 'PARTIAL':'OK',
            'message'     => 'windows_recovered='.$winRecovered
        ]);

        return $this->response->setJSON([
            'ok' => $errs===0,
            'pages'=>$pages,'ins'=>$ins,'upd'=>$upd,'inact'=>$inact,'errs'=>$errs,
            'windows_recovered'=>$winRecovered
        ]);
    }

    /**
     * Cuando una página base (size=pageSize) falla, subdivide la misma ventana
     * en tamaños más pequeños para rescatar ítems sanos.
     * Corta rápido si size=1 falla repetidamente.
     */
    private function recoverFailedWindow(string $prefix, int $basePage, string $runMark, \App\Models\PreciosSyncModel $sync): array
    {
        $baseSize = $this->pageSize;      // usa el tamaño real configurado
        $sizes = [100, 20, 1];            // granularidades de recuperación
        $ins=0; $upd=0; $errs=0; $pages=0;

        foreach ($sizes as $s) {
            // calcular el rango de subpáginas que cubren la ventana base
            $start = intdiv($basePage * $baseSize, $s);
            $end   = intdiv(($basePage+1) * $baseSize - 1, $s);

            $hardFails = 0; // fallos “duros” consecutivos en este tamaño
            for ($sp = $start; $sp <= $end; $sp++) {
                try {
                    $pd = $this->fetchPageExact($prefix, $sp, $s);
                    $data = $pd['data'] ?? [];
                    if (empty($data)) continue;

                    [$rows,$stocks] = $this->mapPage($data, $runMark);
                    $res = $sync->syncPage($rows, $stocks);
                    if (!empty($res['ok'])) { $ins += $res['inserted']??0; $upd += $res['updated']??0; }
                    else { $errs++; }

                    $pages++;
                    $hardFails = 0; // se resetea si hubo éxito
                } catch (\Throwable $e2) {
                    log_message('error', "RECOVERY fallo [$prefix] basePage=$basePage size=$s subPage=$sp: ".$e2->getMessage());
                    $errs++; $hardFails++;
                    if ($s === 1 && $hardFails >= 5) { // corte rápido en size=1
                        log_message('error', "RECOVERY abortado [$prefix] basePage=$basePage size=1 por fallos consecutivos");
                        break;
                    }
                }
            }
        }

        return compact('ins','upd','errs','pages');
    }

    /** Llama al proveedor con un size exacto (para recovery por subpáginas) */
    private function fetchPageExact(string $searchValue, int $page, int $size): array
    {
        $endpoint = $this->baseUri . 'pricing/search-products';
        $query    = http_build_query([
            'searchValue' => $searchValue,
            'page'        => $page,
            'size'        => $size,
        ], '', '&', PHP_QUERY_RFC3986);

        $headers = [
            'Accept: application/json',
            'Content-Type: application/json',
            'Authorization' => $this->authRaw,
        ];

        $resp = $this->curlGet($endpoint.'?'.$query, $headers, $this->timeout);
        $arr  = json_decode($resp, true);
        if (!is_array($arr)) throw new \RuntimeException('Respuesta inválida (recovery) page='.$page.' size='.$size);
        return $arr;
    }
}
