<?php
namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\ListaPreciosModel;
use App\Models\PriceListSyncLogModel;

class SincronizacionListas extends BaseController
{
    private $baseUri;
    private $authRaw;
    private $timeout = 60; // segundos

    public function __construct()
    {
        $this->baseUri = rtrim(getenv('precios.base_uri') ?: 'https://int.api.megaprofer.com/', '/') . '/';
        $this->authRaw = getenv('precios.auth_raw') ?: '';
        $this->timeout = (int) (getenv('precios.timeout') ?: $this->timeout);
    }

    /**
     * GET /sincronizar/price-lists
     * - Descarga listas desde /pricing/price-lists
     * - Upsert por lpr_ref_id (referenceId)
     * - Pruning: marca active=false a las que no vengan esta corrida
     * - Log en tbl_price_list_sync_log
     */
    public function price_lists()
    {
        @set_time_limit(0);

        $log = new PriceListSyncLogModel();
        $logId = $log->start(true);

        $m = new ListaPreciosModel();

        try {
            // 1) Llamar API
            $url = $this->baseUri . 'pricing/price-lists';
            $headers = [
                'Accept: application/json',
                'Content-Type: application/json',
                'Authorization: ' . $this->authRaw,
                'User-Agent: PriceListsSync/1.0',
            ];
            $body = $this->curlGet($url, $headers, $this->timeout);

            $arr = json_decode($body, true);
            if (!is_array($arr)) {
                throw new \RuntimeException('Respuesta inválida del API de price-lists');
            }

            // 2) Normalizar items
            $items = [];
            $refIds = [];
            foreach ($arr as $row) {
                if (!is_array($row)) continue;

                $refId = (string)($row['referenceId'] ?? '');
                if ($refId === '') continue;

                $items[] = [
                    'refId'  => $refId,
                    'name'   => (string)($row['name'] ?? ''),
                    'extId'  => (string)($row['id'] ?? ''),
                    // active puede venir true/false/null; lo guardamos tal cual
                    'active' => array_key_exists('active', $row) ? $row['active'] : null,
                ];
                $refIds[] = $refId;
            }

            // 3) Upsert + pruning
            $up = $m->upsertManyFromApi($items);
            $pruned = $m->pruneMissing($refIds);

            // 4) Log y respuesta
            $log->finish($logId, [
                'ok'        => true,
                'message'   => 'OK',
                'inserted'  => $up['inserted'] ?? 0,
                'updated'   => $up['updated'] ?? 0,
                'pruned'    => $pruned,
                'total_api' => count($refIds),
            ]);

            return $this->response->setJSON([
                'ok'       => true,
                'inserted' => $up['inserted'] ?? 0,
                'updated'  => $up['updated'] ?? 0,
                'pruned'   => $pruned,
                'total'    => count($refIds),
            ]);
        } catch (\Throwable $e) {
            // Log de error con ok=false (boolean)
            try {
                $log->finish($logId, [
                    'ok'      => false,
                    'message' => $e->getMessage(),
                ]);
            } catch (\Throwable $ignore) {
                // Evita que un fallo de logging tape el error original
            }

            return $this->response
                ->setStatusCode(500)
                ->setJSON(['ok' => false, 'error' => $e->getMessage()]);
        }
    }

    /* ====================== HTTP helper ====================== */

    private function curlGet($url, $headers, $timeout)
    {
        $ch = curl_init();
        if ($ch === false) {
            throw new \RuntimeException('No se pudo inicializar cURL');
        }

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_CONNECTTIMEOUT => $timeout,
            CURLOPT_TIMEOUT        => $timeout,
            CURLOPT_HTTPHEADER     => $headers,
        ]);

        $body = curl_exec($ch);
        if ($body === false) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new \RuntimeException('Error cURL: ' . $err);
        }

        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($code < 200 || $code >= 300) {
            throw new \RuntimeException("HTTP $code: $body");
        }

        return $body;
    }
}
