<?php namespace App\Libraries;

use CodeIgniter\Database\BaseConnection;

class FieldResolver
{
    /** @var BaseConnection */
    protected $db;

    /**
     * Whitelist de tablas/columnas que se pueden leer.
     * Agrega aquí cualquier fuente nueva que vaya a usar el catálogo.
     */
    protected $allow = [
        'tbl_producto' => [
            'pro_id','pro_codigo','pro_descripcion','pro_familia','pro_gancho','pro_categorizacion',
            'pro_proveedor_habitual','pro_bussiness','pro_unidad_nombre',
            'pro_cost','pro_payed_price','pro_price_cost','pro_reference_id'
        ],
        'vw_pdv_last_price' => [
            'price','base_price','normal_price','wholesaler_price','special_price','administrator_price','margin_price_pdv'
        ],
        'tbl_parametros_producto' => [
            'ppr_margen_hoy',
            'ppr_desc_normal','ppr_desc_limite','ppr_desc_admin','ppr_desc_mayorista','ppr_desc_especial',
            'ppr_margen_normal','ppr_margen_limite','ppr_margen_admin','ppr_margen_mayorista','ppr_margen_especial',
            'ppr_sentido_normal','ppr_sentido_limite','ppr_sentido_admin','ppr_sentido_mayorista','ppr_sentido_especial',
            'ppr_activo','ppr_updated_at'
        ],
        'tbl_producto_stock' => [
            'stock','updated_at','warehouse_name','pro_reference_id'
        ],
        'vw_descuento_aplicado_ui'        => ['dea_descuento','pro_codigo'],
        'tbl_descuento_base'            => ['des_descuento','pro_codigo'],
        'vw_margen_hoy_ui'          => ['mho_margen','pro_codigo'],
        'vw_incremento_precio_hoy_ui'     => ['iph_incremento_ui','pro_codigo'],
        'vw_incremento_mejores_precios_ui'=> ['imp_incremento_ui','pro_codigo'],
        'vw_utilidad_esp_aut_ui'=> ['uea_utilidad','pro_codigo']
    ];

    /** Caches por-request */
    protected array $productCache = [];     // pro_id => fila tbl_producto
    protected array $paramsCache  = [];     // pro_codigo => fila tbl_parametros_producto
    protected array $keyColumnCache = [];   // table => key column name

    public function __construct()
    {
        $this->db = \Config\Database::connect('postgres');
    }

    /* ============================================================
     *    MÉTODOS DE GESTIÓN DEL ALLOWLIST
     * ============================================================ */

    /** Agrega una tabla (y opcionalmente columnas) al allowlist. */
    public function addAllowlistTable(string $table, array $columns = []): self
    {
        $table = trim($table);
        if ($table === '') return $this;

        if (!isset($this->allow[$table])) {
            $this->allow[$table] = [];
        }
        if (!empty($columns)) {
            $cols = array_map('strval', $columns);
            $this->allow[$table] = array_values(array_unique(array_merge($this->allow[$table], $cols)));
        }
        return $this;
    }

    /** Agrega columnas al allowlist para una tabla existente o nueva. */
    public function addAllowlistColumns(string $table, array $columns): self
    {
        return $this->addAllowlistTable($table, $columns);
    }

    /** Verifica si (tabla,columna) está permitido. */
    public function isAllowed(string $table, string $column): bool
    {
        return isset($this->allow[$table]) && in_array($column, $this->allow[$table], true);
    }

    /* ============================================================
     *                    API PRINCIPAL
     * ============================================================ */

    /**
     * Resuelve el valor de un campo global (param/lectura/fórmula)
     */
    public function resolveField(array $campoGlobal, array $prod, ?int $pdvId, array &$ctx)
    {
        $name   = $campoGlobal['cam_nombre'];
        if (array_key_exists($name, $ctx)) return $ctx[$name];

        $origen = $campoGlobal['cam_origen'] ?? ($campoGlobal['origen'] ?? 'lectura');
        $tipo   = $campoGlobal['cam_tipo']   ?? ($campoGlobal['tipo']   ?? 'text');

        // PARAM
        if ($origen === 'param') {
            $val = $campoGlobal['cam_formula'] ?? ($campoGlobal['formula'] ?? null);
            return $ctx[$name] = $this->cast($val, $tipo);
        }

        // LECTURA
        if ($origen === 'lectura') {
            $src = $campoGlobal['cam_source'] ?? ($campoGlobal['source'] ?? null);
            $src = is_array($src) ? $src : (is_string($src) && $src !== '' ? (json_decode($src, true) ?: []) : []);
            if (!$src) { $ctx[$name] = null; return null; }
            $val = $this->readFromSource($src, $prod, $pdvId);
            return $ctx[$name] = $this->cast($val, $tipo);
        }

        // FÓRMULA
        if ($origen === 'formula') {
            $deps = $campoGlobal['cam_depends'] ?? ($campoGlobal['depends'] ?? []);
            if (!is_array($deps)) $deps = json_decode((string)$deps, true) ?: [];

            // resolver dependencias
            foreach ($deps as $d) {
                if (!array_key_exists($d, $ctx)) {
                    $depDef = $this->findCampoGlobal($d);
                    if ($depDef) {
                        $ctx[$d] = $this->resolveField($depDef, $prod, $pdvId, $ctx);
                    } else {
                        $ctx[$d] = null;
                    }
                }
            }
            $depVals = [];
            foreach ($deps as $d) $depVals[$d] = $ctx[$d] ?? null;

            $expr = (string)($campoGlobal['cam_formula'] ?? ($campoGlobal['formula'] ?? ''));
            $val  = $expr !== '' ? $this->evalFormula($expr, $depVals, $tipo) : null;
            return $ctx[$name] = $this->cast($val, $tipo);
        }

        return $ctx[$name] = null;
    }

    /**
     * Lee desde una fuente declarada: {"table":"...","column":"..."}
     */
    public function readFromSource(array $src, array $prod, ?int $pdvId = null)
    {
        $table = $src['table'] ?? null;
        $col   = $src['column'] ?? null;
        if (!$table || !$col) return null;
        if (!$this->isAllowed($table, $col)) return null;

        // tbl_producto — intenta del SELECT y si no, cache
        if ($table === 'tbl_producto') {
            if (array_key_exists($col, $prod)) return $prod[$col] ?? null;

            $proId = $prod['pro_id'] ?? null;
            if ($proId) {
                $row = $this->getProductRowById((int)$proId);
                return $row[$col] ?? null;
            }
            $codigo = $prod['pro_codigo'] ?? null;
            if ($codigo) {
                $row = $this->db->table('public.tbl_producto')->where('pro_codigo',$codigo)->get()->getRowArray();
                if ($row && isset($row['pro_id'])) $this->productCache[(int)$row['pro_id']] = $row;
                return $row[$col] ?? null;
            }
            return null;
        }

        // vw_pdv_last_price — por clave dinámica
        if ($table === 'vw_pdv_last_price') {
    $codigo = $prod['pro_codigo'] ?? $this->readFromSource(
        ['table' => 'tbl_producto', 'column' => 'pro_codigo'],
        $prod,
        $pdvId
    );
    if (!$codigo) return null;

    try {
        $row = $this->db->table('public.vw_pdv_last_price')
            ->select($col . ' AS v')
            ->where('pro_codigo', $codigo)  // <-- filtramos directamente
            ->get()
            ->getRowArray();

        if (!$row) {
            log_message('error', "[FR] vw_pdv_last_price no devolvió valor para {$codigo}");
        } else {
            log_message('debug', "[FR] vw_pdv_last_price {$codigo} -> " . json_encode($row));
        }

        return $row['v'] ?? null;
    } catch (\Throwable $e) {
        log_message('error', 'vw_pdv_last_price query failed: ' . $e->getMessage());
        return null;
    }
}



        // tbl_parametros_producto — por pro_codigo con cache
        if ($table === 'tbl_parametros_producto') {
            $codigo = $prod['pro_codigo'] ?? null;
            if (!$codigo) $codigo = $this->readFromSource(['table'=>'tbl_producto','column'=>'pro_codigo'], $prod, $pdvId);
            if (!$codigo) return null;

            $row = $this->getParamsRowByCodigo($codigo);
            return $row[$col] ?? null;
        }

        // tbl_producto_stock — por pro_reference_id más reciente
        if ($table === 'tbl_producto_stock') {
            $ref = $prod['pro_reference_id'] ?? null;
            if (!$ref) $ref = $this->readFromSource(['table'=>'tbl_producto','column'=>'pro_reference_id'], $prod, $pdvId);
            if (!$ref) return null;

            $row = $this->db->table('public.tbl_producto_stock')
                ->select($col.' AS v')
                ->where('pro_reference_id',$ref)
                ->orderBy('updated_at','desc')
                ->get()->getRowArray();
            return $row['v'] ?? null;
        }

        // Tablas/vistas simples por clave dinámica
        if (in_array($table, [
            'vw_descuento_aplicado_ui','tbl_descuento_base',
            'vw_margen_hoy_ui','vw_incremento_precio_hoy_ui','vw_incremento_mejores_precios_ui','vw_utilidad_esp_aut_ui'
        ], true)) {
            $codigo = $prod['pro_codigo'] ?? null;
            if (!$codigo) $codigo = $this->readFromSource(['table'=>'tbl_producto','column'=>'pro_codigo'], $prod, $pdvId);
            if (!$codigo) return null;

            $keyCol = $this->resolveKeyColumn($table);

            try {
                $row = $this->db->table('public.'.$table)
                    ->select($col.' AS v')
                    ->where($keyCol, $codigo)
                    ->get()->getRowArray();
                return $row['v'] ?? null;
            } catch (\Throwable $e) {
                log_message('error', $table.' query failed: '.$e->getMessage());
                return null;
            }
        }

        // Fallbacks encadenados (opcional)
        if (!empty($src['fallbacks']) && is_array($src['fallbacks'])) {
            foreach ($src['fallbacks'] as $f) {
                if (strpos($f,'const:') === 0) {
                    $const = substr($f,6);
                    return is_numeric($const) ? (float)$const : $const;
                }
                if (strpos($f,'.') !== false) {
                    [$t,$c] = explode('.', $f, 2);
                    if ($this->isAllowed($t, $c)) {
                        return $this->readFromSource(['table'=>$t,'column'=>$c], $prod, $pdvId);
                    }
                }
            }
        }
        return null;
    }

    /**
     * Evalúa una expresión SQL-safe con binds posicionales y funciones/keywords whitelisted.
     * Normaliza números con coma y permite saltos de línea / %
     */
    public function evalFormula(string $expr, array $vars, string $returnType = 'number')
    {
        // decide el CAST de variables según el tipo esperado
        $castByType = [
            'number'  => 'numeric',
            'money'   => 'numeric',
            'percent' => 'numeric',
            'text'    => 'text',
            'date'    => 'date',
            'bool'    => 'boolean',
        ];
        $pgCast = $castByType[strtolower($returnType)] ?? 'numeric';

        $binds = [];

        // funciones/palabras permitidas (en MAYÚSCULAS)
        $allowed = [
            'COALESCE','NULLIF','ABS','ROUND','FLOOR','CEIL','CEILING','POWER','LEAST','GREATEST',
            'CASE','WHEN','THEN','ELSE','END',
            'AND','OR','NOT','IS','NULL','TRUE','FALSE','IN','LIKE','ILIKE','BETWEEN'
        ];

        // Reemplazo seguro (normaliza valores numéricos que llegan como texto con coma)
        $safe = preg_replace_callback(
            '/\b[a-zA-Z_][a-zA-Z0-9_]*\b/',
            function($m) use ($vars, &$binds, $allowed, $pgCast) {
                $name = $m[0];

                if (array_key_exists($name, $vars)) {
                    $val = $vars[$name];

                    if (is_string($val)) {
                        // quita espacios duros y comunes
                        $val = str_replace(["\xC2\xA0",' '], '', $val);
                        // si parece número con coma/punto
                        if (preg_match('/^-?[0-9\.,]+$/', $val)) {
                            // elimina posibles separadores de miles "." y convierte coma decimal a "."
                            $val = str_replace('.', '', $val);
                            $val = str_replace(',', '.', $val);
                        }
                    }
                    $binds[] = $val;
                    return '(CAST(? AS '.$pgCast.'))';
                }

                $upper = strtoupper($name);
                if (in_array($upper, $allowed, true)) return $upper;

                return $name;
            },
            $expr
        );

        // Validación de caracteres (permite saltos de línea, % y comillas dobles)
        if (!preg_match("/^[0-9 \t\r\n\+\-\*\/%\.\,\(\)A-Za-z_\?\"'<>=:]+$/", $safe)) {
            throw new \RuntimeException('Fórmula contiene caracteres no permitidos');
        }

        $q = $this->db->query('SELECT ('.$safe.') AS v', $binds);
        $row = $q->getRowArray();
        return $row['v'] ?? null;
    }

    /** Casting con redondeo a 4 decimales en números/porcentajes */
    private function cast($v, string $tipo)
    {
        if ($v === null) return null;
        switch (strtolower($tipo)) {
            case 'number':
            case 'money':
            case 'percent':
                if (is_numeric($v)) return round((float)$v, 4);
                if (is_string($v)) {
                    $vv = str_replace(["\xC2\xA0",' '], '', $v);
                    $vv = str_replace('.', '', $vv);
                    $vv = str_replace(',', '.', $vv);
                    return is_numeric($vv) ? round((float)$vv, 4) : null;
                }
                return null;
            case 'bool':
                return in_array($v, [1,'1',true,'true','t','on','TRUE'], true);
            case 'date':
                return (string)$v;
            default:
                return (string)$v;
        }
    }

    /** Busca definición de un campo global por nombre (para resolver dependencias) */
    protected function findCampoGlobal(string $camNombre): ?array
    {
        $row = $this->db->table('public.tbl_campo_global')
            ->where('cam_nombre', $camNombre)
            ->get()->getRowArray();
        return $row ?: null;
    }

    /** Cache: fila completa de tbl_producto por pro_id */
    protected function getProductRowById(int $proId): array
    {
        if (isset($this->productCache[$proId])) return $this->productCache[$proId];
        try {
            $row = $this->db->table('public.tbl_producto')->where('pro_id',$proId)->get()->getRowArray() ?: [];
            return $this->productCache[$proId] = $row;
        } catch (\Throwable $e) {
            log_message('error','getProductRowById failed: '.$e->getMessage());
            return [];
        }
    }

    /** Cache: fila de parámetros por pro_codigo */
    protected function getParamsRowByCodigo(string $proCodigo): array
    {
        if (isset($this->paramsCache[$proCodigo])) return $this->paramsCache[$proCodigo];
        try {
            $row = $this->db->table('public.tbl_parametros_producto')
                ->where('pro_codigo', $proCodigo)
                ->where('COALESCE(ppr_activo,true) = true', null, false)
                ->orderBy('ppr_updated_at','desc')
                ->get()->getRowArray() ?: [];
            return $this->paramsCache[$proCodigo] = $row;
        } catch (\Throwable $e) {
            log_message('error','getParamsRowByCodigo failed: '.$e->getMessage());
            return [];
        }
    }

    /**
     * Devuelve el nombre de columna a usar como clave en una tabla/vista.
     * Revisa candidatos comunes y, si no, introspecciona information_schema.
     */
    protected function resolveKeyColumn(string $table): string
    {
        if (isset($this->keyColumnCache[$table])) return $this->keyColumnCache[$table];

        // candidatos en orden de preferencia
        $candidates = ['pro_codigo','codigo','product_code','sku','pro_id','id'];

        try {
            $cols = $this->db->query("
                SELECT column_name
                FROM information_schema.columns
                WHERE table_schema IN ('public')
                  AND table_name = ?
            ", [$table])->getResultArray();

            $have = array_map(fn($r) => strtolower($r['column_name']), $cols);

            foreach ($candidates as $c) {
                if (in_array(strtolower($c), $have, true)) {
                    return $this->keyColumnCache[$table] = $c;
                }
            }
            if (!empty($have)) {
                return $this->keyColumnCache[$table] = $have[0]; // fallback
            }
        } catch (\Throwable $e) {
            // silencioso: caemos a pro_codigo
        }
        return $this->keyColumnCache[$table] = 'pro_codigo';
    }
}
