<?php
namespace App\Controllers;

use App\Models\ProductoModel;
use App\Models\ProductoStockModel;
use App\Models\SyncLogProductoModel;
use App\Models\SyncCursorProductoModel;

class Sincronizacion extends BaseController
{
    private string $baseUri;
    private string $authRaw;
    private int    $pageSize   = 500; // puedes override con ?size=
    private int    $maxRetries = 2;
    private int    $timeout    = 60;

    public function __construct()
    {
        $this->baseUri = rtrim(getenv('precios.base_uri') ?: 'https://int.api.megaprofer.com/', '/') . '/';
        $this->authRaw = getenv('precios.auth_raw') ?: 'rapidito-int;N4RaVsn|&BaXLTIk]1';
        $this->timeout = (int)(getenv('precios.timeout') ?: $this->timeout);
    }

    /** Paso 0: crear corrida, procesar página 0 y devolver run_id */
    public function iniciar()
    {
        @set_time_limit(0);

        $qsSize = (int)($this->request->getGet('size') ?? 0);
        if ($qsSize > 0) $this->pageSize = $qsSize;

        $mLog = new SyncLogProductoModel();
        $mCur = new SyncCursorProductoModel();

        $startedAt = date('Y-m-d H:i:s');
        $logId = $mLog->insert([
            'started_at' => $startedAt,
            'page_size'  => $this->pageSize,
            'status'     => 'RUNNING',
            'message'    => 'Corrida por pasos (HTTP)'
        ], true);

        try {
            // Obtener totales y procesar página 0
            $first   = $this->fetchPageWithRetry(0);
            $totPg   = (int)($first['totalPages'] ?? 0);
            $totElem = (int)($first['totalElements'] ?? 0);

            [$rows, $stocks] = $this->mapPage($first['data'] ?? [], $startedAt);
            $db = \Config\Database::connect('postgres');
            $db->transStart();
            $mProd  = new ProductoModel();
            $mStock = new ProductoStockModel();
            $res    = $mProd->upsertBatchWithHash($rows);
            $ins0   = $res['inserted'] ?? 0;
            $upd0   = $res['updated'] ?? 0;
            $mStock->upsertStocks($stocks);
            $db->transComplete();
            $err0 = $db->transStatus() ? 0 : 1;

            // Crear cursor en RUNNING (página 0 ya procesada)
            $runId = $mCur->insert([
                'started_at'   => $startedAt,
                'page_size'    => $this->pageSize,
                'total_pages'  => $totPg,
                'current_page' => 0,
                'inserted'     => $ins0,
                'updated'      => $upd0,
                'errors'       => $err0,
                'log_id'       => $logId,
                'status'       => 'RUNNING',
            ], true);

            // actualizar log con totales
            $mLog->update($logId, [
                'total_pages'    => $totPg,
                'total_elements' => $totElem,
            ]);

            return $this->response->setJSON([
                'ok'          => true,
                'run_id'      => (int)$runId,
                'total_pages' => $totPg,
                'processed'   => 1,   // ya hicimos page 0
                'inserted'    => $ins0,
                'updated'     => $upd0,
                'errors'      => $err0
            ]);

        } catch (\Throwable $e) {
            $mLog->update($logId, [
                'finished_at' => date('Y-m-d H:i:s'),
                'status'      => 'ERROR',
                'message'     => $e->getMessage()
            ]);
            return $this->response->setStatusCode(500)->setJSON(['ok'=>false,'error'=>$e->getMessage()]);
        }
    }

    /** Paso N: procesa las siguientes ?pages (default 1) páginas */
    public function continuar($runId)
    {
        @set_time_limit(0);

        $pagesPerCall = max(1, (int)($this->request->getGet('pages') ?? 1));

        $mCur = new SyncCursorProductoModel();
        $mLog = new SyncLogProductoModel();
        $mProd  = new ProductoModel();
        $mStock = new ProductoStockModel();

        $cur = $mCur->find((int)$runId);
        if (!$cur) return $this->response->setStatusCode(404)->setJSON(['ok'=>false,'error'=>'run_id no encontrado']);
        if ($cur['status'] !== 'RUNNING') {
            return $this->response->setJSON(['ok'=>true,'done'=>true] + $cur);
        }

        $pageSize   = (int)$cur['page_size'];
        $totalPages = (int)$cur['total_pages'];
        $current    = (int)$cur['current_page']; // última procesada
        $errs       = (int)$cur['errors'];
        $insTot     = (int)$cur['inserted'];
        $updTot     = (int)$cur['updated'];
        $logId      = (int)$cur['log_id'];
        $runMark    = $cur['started_at'];

        $processedThisCall = 0;

        try {
            for ($i = 0; $i < $pagesPerCall; $i++) {
                $next = $current + 1;
                if ($next >= $totalPages) break;

                try {
                    $page = $this->fetchPageExactWithRetry($next, $pageSize); // primero con size base
                    $data = $page['data'] ?? [];
                    if (empty($data)) { $current = $next; $processedThisCall++; continue; }

                    [$rows, $stocks] = $this->mapPage($data, $runMark);
                    $db = \Config\Database::connect('postgres');
                    $db->transStart();
                    $res = $mProd->upsertBatchWithHash($rows);
                    $insTot += $res['inserted'] ?? 0;
                    $updTot += $res['updated'] ?? 0;
                    $mStock->upsertStocks($stocks);
                    $db->transComplete();
                    if (!$db->transStatus()) { $errs++; }

                    $current = $next;
                    $processedThisCall++;

                } catch (\Throwable $ePage) {
                    // fallback de tamaño para esta página puntual
                    [$offset, $winSize] = $this->pageWindow($next, $pageSize);
                    $window = $this->fetchWindowWithFallback($offset, $winSize, [250,100,50,20]);
                    if (!$window['ok']) {
                        $errs++;
                        log_message('error', 'Sync productos (run '.$runId.'): ventana offset='.$offset.' size='.$winSize.' fallo. '.$ePage->getMessage());
                        // marcamos como “saltada” pero avanzamos el puntero para no bloquearnos
                        $current = $next;
                        $processedThisCall++;
                        continue;
                    }
                    foreach ($window['pages'] as $chunk) {
                        $sub = $chunk['arr'];
                        $data = $sub['data'] ?? [];
                        if (empty($data)) continue;
                        $db = \Config\Database::connect('postgres');
                        $db->transStart();
                        [$rows, $stocks] = $this->mapPage($data, $runMark);
                        $res = $mProd->upsertBatchWithHash($rows);
                        $insTot += $res['inserted'] ?? 0;
                        $updTot += $res['updated'] ?? 0;
                        $mStock->upsertStocks($stocks);
                        $db->transComplete();
                        if (!$db->transStatus()) { $errs++; }
                    }
                    $current = $next;
                }
            }

            // actualizar cursor
            $mCur->update($runId, [
                'current_page' => $current,
                'inserted'     => $insTot,
                'updated'      => $updTot,
                'errors'       => $errs,
            ]);
            // actualizar log parcial
            $mLog->update($logId, [
                'inserted' => $insTot,
                'updated'  => $updTot,
                'errors'   => $errs,
            ]);

            // ¿terminó?
            if ($current >= $totalPages - 1) {
                // inactivar solo si no hubo errores
                $inact = 0;
                if ($errs === 0) {
                    $inact = $mProd->inactivateNotSeenSince($runMark);
                }
                $status = $errs === 0 ? 'OK' : 'PARTIAL';
                $mCur->update($runId, [
                    'finished_at' => date('Y-m-d H:i:s'),
                    'status'      => $status
                ]);
                $mLog->update($logId, [
                    'finished_at'  => date('Y-m-d H:i:s'),
                    'inactivated'  => $inact,
                    'status'       => $status,
                    'message'      => $status === 'OK' ? 'Sync completa por pasos' : 'Sync parcial por pasos'
                ]);

                return $this->response->setJSON([
                    'ok'=>true,'done'=>true,
                    'run_id'      => (int)$runId,
                    'currentPage' => $current,
                    'totalPages'  => $totalPages,
                    'inserted'    => $insTot,
                    'updated'     => $updTot,
                    'errors'      => $errs,
                    'inactivated' => $inact,
                    'status'      => $status
                ]);
            }

            return $this->response->setJSON([
                'ok'=>true,'done'=>false,
                'run_id'      => (int)$runId,
                'currentPage' => $current,
                'processedThisCall' => $processedThisCall,
                'totalPages'  => $totalPages,
                'inserted'    => $insTot,
                'updated'     => $updTot,
                'errors'      => $errs
            ]);

        } catch (\Throwable $e) {
            $mCur->update($runId, ['status'=>'ERROR','finished_at'=>date('Y-m-d H:i:s')]);
            if ($logId) $mLog->update($logId, ['status'=>'ERROR','finished_at'=>date('Y-m-d H:i:s'),'message'=>$e->getMessage()]);
            return $this->response->setStatusCode(500)->setJSON(['ok'=>false,'error'=>$e->getMessage()]);
        }
    }

    /** Ver estado */
    public function estado($runId)
    {
        $mCur = new SyncCursorProductoModel();
        $cur = $mCur->find((int)$runId);
        if (!$cur) return $this->response->setStatusCode(404)->setJSON(['ok'=>false,'error'=>'run_id no encontrado']);
        return $this->response->setJSON(['ok'=>true] + $cur);
    }

    /* --------------------- Helpers HTTP y fallback --------------------- */

    private function fetchPageWithRetry(int $page): array
    {
        $attempt = 0;
        while (true) {
            try { return $this->fetchPage($page); }
            catch (\Throwable $e) { if (++$attempt > $this->maxRetries) throw $e; sleep($attempt===1?2:5); }
        }
    }
    private function fetchPageExactWithRetry(int $page, int $size): array
    {
        $attempt = 0;
        while (true) {
            try { return $this->fetchPageExact($page, $size); }
            catch (\Throwable $e) { if (++$attempt > $this->maxRetries) throw $e; sleep($attempt===1?2:5); }
        }
    }

    private function fetchPage(int $page): array
    {
        $endpoint = $this->baseUri . 'pricing/search-products';
        $query = http_build_query(['searchValue'=>'','page'=>$page,'size'=>$this->pageSize], '', '&', PHP_QUERY_RFC3986);
        $headers = ['Accept: application/json','Content-Type: application/json','Authorization: '.$this->authRaw];
        $resp = $this->curlGet("$endpoint?$query", $headers, $this->timeout);
        $arr  = json_decode($resp, true);
        if (!is_array($arr)) throw new \RuntimeException('Respuesta inválida page='.$page);
        return $arr;
    }

    private function fetchPageExact(int $page, int $size): array
    {
        $endpoint = $this->baseUri . 'pricing/search-products';
        $query = http_build_query(['searchValue'=>'','page'=>$page,'size'=>$size], '', '&', PHP_QUERY_RFC3986);
        $headers = ['Accept: application/json','Content-Type: application/json','Authorization: '.$this->authRaw];
        $resp = $this->curlGet("$endpoint?$query", $headers, $this->timeout);
        $arr  = json_decode($resp, true);
        if (!is_array($arr)) throw new \RuntimeException('Respuesta inválida exact page='.$page.' size='.$size);
        return $arr;
    }

    private function curlGet(string $url, array $headers, int $timeout): string
    {
        $ch = curl_init();
        if ($ch === false) throw new \RuntimeException('No se pudo inicializar cURL');
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_CONNECTTIMEOUT => $timeout,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_HTTPHEADER => $headers,
            // Descomenta si el INT tiene cert no público:
            // CURLOPT_SSL_VERIFYPEER => false,
            // CURLOPT_SSL_VERIFYHOST => false,
        ]);
        $body = curl_exec($ch);
        if ($body === false) { $err = curl_error($ch); curl_close($ch); throw new \RuntimeException('cURL: '.$err); }
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        if ($code < 200 || $code >= 300) throw new \RuntimeException("HTTP $code: $body");
        return $body;
    }

    private function pageWindow(int $page, int $size): array
    {
        $offset = $page * $size;
        return [$offset, $size];
    }

    private function fetchWindowWithFallback(int $offset, int $windowSize, array $sizesTry): array
    {
        foreach ($sizesTry as $sz) {
            $startPage = intdiv($offset, $sz);
            $endOffset = $offset + $windowSize - 1;
            $endPage   = intdiv($endOffset, $sz);

            $collected = []; $failed = false;
            for ($pp = $startPage; $pp <= $endPage; $pp++) {
                try {
                    $arr = $this->fetchPageExact($pp, $sz);
                    $collected[] = ['page'=>$pp,'size'=>$sz,'arr'=>$arr];
                } catch (\Throwable $e) {
                    $failed = true; break;
                }
            }
            if (!$failed) return ['ok'=>true, 'pages'=>$collected, 'error'=>null];
        }
        return ['ok'=>false, 'pages'=>[], 'error'=>'Fallo persistente'];
    }

    /* --------------------- Mapeo a filas --------------------- */

    private function n6($v) { return $v === null ? null : round((float)$v, 6); }
    private function n4($v) { return $v === null ? null : round((float)$v, 4); }

    private function mapPage(array $data, string $runMark): array
    {
        $rows = []; $stocks = [];
        foreach ($data as $it) {
            $refId   = (string)($it['referenceId'] ?? '');
            $code    = (string)($it['referenceCode'] ?? '');
            $barcode = (string)($it['barcode'] ?? '');
            $name    = (string)($it['name'] ?? '');
            $desc    = (string)($it['description'] ?? '');

            $tax     = $it['taxPresenter'] ?? [];
            $unit    = $it['measureUnitPresenter'] ?? [];
            $subline = $it['productSublinePresenter'] ?? [];
            $familyP = $subline['productFamilyPresenter'] ?? [];
            $mark    = $it['markPresenter'] ?? [];
            $stocksA = $it['stocks'] ?? [];

            $row = [
                'pro_reference_id'        => $refId,
                'pro_codigo'              => $code !== '' ? $code : null,
                'pro_barcode'             => $barcode !== '' ? $barcode : null,
                'pro_descripcion'         => $name ?? '',
                'pro_detalle'             => $desc ?? '',
                'pro_familia'             => (string)($familyP['name'] ?? ($it['family'] ?? '')),
                'pro_sublinea'            => (string)($subline['name'] ?? ''),
                'pro_marca'               => (string)($mark['name'] ?? ''),
                'pro_proveedor_habitual'  => (string)($it['provider'] ?? ''),
                'pro_categorizacion'      => (string)($it['categorization'] ?? ''),
                'pro_unidad_nombre'       => (string)($unit['name'] ?? ''),
                'pro_unidad_simbolo'      => (string)($unit['symbol'] ?? ''),
                'pro_iva_nombre'          => (string)($tax['name'] ?? ''),

                'pro_peso'                => $this->n6($it['weight'] ?? null),
                'pro_iva_valor'           => $this->n4($tax['value'] ?? null),
                'pro_cost'                => $this->n6($it['cost'] ?? null),
                'pro_price_cost'          => $this->n6($it['priceCost'] ?? null),
                'pro_payed_price'         => $this->n6($it['payedPrice'] ?? null),
                'pro_stock'               => $this->n6($it['stock'] ?? null),
                'pro_authorized_discount' => $this->n4($it['authorizedDiscount'] ?? null),

                'pro_activo'              => true,
                'pro_es_pesado'           => $it['isHeavy'] ?? null,
                'pro_es_especial'         => $it['isEspecial'] ?? null,
                'pro_need_validate_stock' => $it['needValidateStock'] ?? null,
                'pro_is_supply'           => $it['isSupply'] ?? null,

                'pro_location'            => (string)($it['location'] ?? ''),
                'pro_updated_api'         => !empty($it['updatedDate']) ? $it['updatedDate'] : null,
                'pro_seen_at'             => $runMark,
                'pro_inactive_at'         => null,
                'pro_json'                => json_encode($it, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
            ];

            foreach (['pro_descripcion','pro_detalle','pro_familia','pro_sublinea','pro_marca',
                      'pro_proveedor_habitual','pro_categorizacion','pro_unidad_nombre',
                      'pro_unidad_simbolo','pro_iva_nombre','pro_location'] as $k) {
                if ($row[$k] === null) $row[$k] = '';
            }

            $hashBase = [
                $row['pro_codigo'],$row['pro_barcode'],$row['pro_descripcion'],$row['pro_detalle'],
                $row['pro_familia'],$row['pro_sublinea'],$row['pro_marca'],$row['pro_proveedor_habitual'],
                $row['pro_categorizacion'],$row['pro_unidad_nombre'],$row['pro_unidad_simbolo'],$row['pro_iva_nombre'],
                $row['pro_peso'],$row['pro_iva_valor'],$row['pro_cost'],$row['pro_price_cost'],$row['pro_payed_price'],
                $row['pro_stock'],$row['pro_authorized_discount'],$row['pro_es_pesado'],$row['pro_es_especial'],
                $row['pro_need_validate_stock'],$row['pro_is_supply'],$row['pro_location'],$row['pro_updated_api']
            ];
            $row['pro_hash'] = sha1(json_encode($hashBase));

            $rows[] = $row;

            if (is_array($stocksA)) {
                foreach ($stocksA as $st) {
                    $wName = (string)($st['warehouseName'] ?? '');
                    if ($wName === '') continue;
                    $stocks[] = [
                        'pro_reference_id' => $refId,
                        'warehouse_name'   => $wName,
                        'stock'            => $this->n6($st['stock'] ?? 0),
                    ];
                }
            }
        }
        return [$rows, $stocks];
    }
}
