<?php
namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\MatrizModel;
use App\Models\CampoGlobalModel;
use App\Models\MatrizCampoRefModel;
use App\Models\MatrizDatoModel;
use App\Libraries\FieldResolver;
use App\Models\MatrizVersionModel;

class Matrices2Controller extends BaseController
{
    protected $db;
    protected $matriz;
    protected $campos;
    protected $mcr;
    protected $mad;
    protected $resolver;

    public function __construct()
    {
        $this->db       = \Config\Database::connect('postgres');
        $this->matriz   = new MatrizModel();
        $this->campos   = new CampoGlobalModel();
        $this->mcr      = new MatrizCampoRefModel();
        $this->mad      = new MatrizDatoModel();
        $this->resolver = new FieldResolver();
    }

    // ===== UI: listado de matrices
    public function index()
    {
        $mats = $this->matriz->orderBy('mat_id','asc')->findAll();
        echo view('layouts/header');
        echo view('layouts/aside');
        echo view('matrices2/index', ['matrices'=>$mats]);
        echo view('layouts/footer');
    }

    // ===== API: crear matriz
    public function create()
    {
        $p = $this->request->getPost();
        $data = [
            'mat_tipo'   => trim($p['mat_tipo'] ?? 'GENERICA'),
            'mat_nombre' => trim($p['mat_nombre'] ?? ''),
            'mat_estado' => 1,
            'created_at' => date('c'),
            'updated_at' => date('c'),
        ];
        if ($data['mat_nombre']==='') {
            return $this->response->setStatusCode(422)->setJSON(['ok'=>false,'msg'=>'Nombre requerido']);
        }
        $this->matriz->insert($data);
        return $this->response->setJSON(['ok'=>true,'id'=>$this->matriz->getInsertID()]);
    }

    // ===== UI: builder (seleccin/orden/visibilidad de campos)
    public function builder($matId)
    {
        $mat = $this->matriz->find((int)$matId);
        if (!$mat) return redirect()->to('/matrices2');

        $all   = $this->campos->listActivos();
        $mrefs = $this->mcr->forMatriz((int)$matId);

        echo view('layouts/header');
        echo view('layouts/aside');
        echo view('matrices2/builder', [
            'mat'    => $mat,
            'campos' => $all,
            'refs'   => $mrefs
        ]);
        echo view('layouts/footer');
    }

    // ===== API: agregar campos a la matriz
    public function addFields($matId)
    {
        $ids = $this->request->getPost('cam_ids') ?? [];
        if (is_string($ids)) $ids = explode(',', $ids);
        $ids = array_map('intval', array_filter($ids));
        if (!$ids) {
            return $this->response->setStatusCode(422)->setJSON(['ok'=>false,'msg'=>'cam_ids vaco']);
        }

        $this->mcr->upsertSet((int)$matId, $ids);
        return $this->response->setJSON(['ok'=>true]);
    }

    // ===== API: actualizar overrides (orden, visible, ttulo, formato, color, editable)
    public function updateRef($mcrId)
    {
        $p = $this->request->getPost();
        $data = [
            'mcr_titulo'  => $p['mcr_titulo'] ?? null,
            'mcr_formato' => $p['mcr_formato'] ?? null,
            'mcr_color'   => $p['mcr_color'] ?? null,
            'mcr_orden'   => ($p['mcr_orden'] === '' ? null : (int)$p['mcr_orden']),
            'mcr_visible' => isset($p['mcr_visible']) ? (int)$p['mcr_visible'] : 1,
        ];
        if (array_key_exists('mcr_editable', $p)) {
            $data['mcr_editable'] = ($p['mcr_editable'] === '' ? null : (int)$p['mcr_editable']);
        }

        $this->mcr->update((int)$mcrId, $data);
        return $this->response->setJSON(['ok'=>true]);
    }

    // ===== API: quitar un campo de la matriz
    public function removeRef($mcrId)
    {
        $this->mcr->delete((int)$mcrId);
        return $this->response->setJSON(['ok'=>true]);
    }

    // ===== UI: grid tipo Excel
    public function grid($matId)
    {
        $mat = $this->matriz->find((int)$matId);
        if (!$mat) return redirect()->to('/matrices2');

        // Campos configurados (en orden) + editable resuelto (mcr > cg)
        $campos = $this->db->table('public.tbl_matriz_campo_ref mcr')
            ->select('mcr.mcr_id, mcr.mcr_titulo, mcr.mcr_formato, mcr.mcr_color, mcr.mcr_orden, mcr.mcr_visible, mcr.mcr_editable,
                      cg.cam_id, cg.cam_nombre, cg.cam_titulo, cg.cam_tipo, cg.cam_origen, cg.cam_source, cg.cam_editable AS cg_editable')
            ->join('public.tbl_campo_global cg','cg.cam_id = mcr.cam_id','inner')
            ->where('mcr.mat_id', (int)$matId)
            ->orderBy('mcr.mcr_orden','asc')
            ->get()->getResultArray();

        foreach ($campos as &$c) {
            $raw = $c['cg_editable'] ?? false;
        
            // normaliza cualquier tipo: bool/int/'t'/'f'/'true'/'false'/etc.
            if (is_bool($raw)) {
                $cg = $raw;
            } elseif (is_int($raw)) {
                $cg = ($raw === 1);
            } else {
                $cg = in_array(strtolower((string)$raw), ['1','t','true','on','yes'], true);
            }
        
            $c['editable'] = $cg && (strtolower((string)$c['cam_origen']) !== 'formula');
        }
        unset($c);



        echo view('layouts/header');
        echo view('layouts/aside');
        echo view('matrices2/grid', ['mat'=>$mat, 'campos'=>$campos]);
        echo view('layouts/footer');
    }

    // ===== API: DataTables server-side para el grid
    public function data($matId)
    {
        // Config general
        @ini_set('max_execution_time', '180');
        @set_time_limit(180);

        $req    = $this->request;
        $draw   = (int)($req->getVar('draw') ?? 1);
        $start  = max(0, (int)($req->getVar('start') ?? 0));
        $len    = (int)($req->getVar('length') ?? 5000);
        $search = trim($req->getVar('search')['value'] ?? '');

        // Sin cap duro: el front trae por chunks; aqu respeta length
        if ($len === 0) $len = 5000;
        if ($len < 0)   $len = 50000; // "todos" con tope seguridad

        // 1) matriz
        $mat = $this->db->table('public.tbl_matriz')
            ->where('mat_id', (int)$matId)->get()->getRowArray();

        // 2) productos base + join con matriz_dato
        $b = $this->db->table('public.tbl_producto p')
            ->select('p.pro_id, p.pro_codigo, p.pro_descripcion, p.pro_familia, p.pro_categorizacion, p.pro_marca, p.pro_reference_id,
                      d.mad_id, d.mad_campos')
            ->join('public.tbl_matriz_dato d', 'd.pro_id = p.pro_id AND d.mat_id = '.(int)$matId, 'left')
            ->where("COALESCE(p.pro_activo,'true') = 'true'", null, false);
            //->where('p.pro_codigo','E05195');

        // 3) filtro por tipo de matriz
        if ($mat && strtoupper($mat['mat_tipo']) === 'PORTAFOLIO') {
            $b->whereIn('p.pro_categorizacion', ['PORTAFOLIO','PORT ESPEC','PORT-PUNTU']);
        } elseif ($mat && strtoupper($mat['mat_tipo']) === 'NO PORTAFOLIO') {
            $b->whereIn('p.pro_categorizacion', ['FUERA-PORT','ADMINISTRA','MKT']);
        }

        // 4) bsqueda
        if ($search !== '') {
            $b->groupStart()
                ->like('p.pro_codigo', $search)
                ->orLike('p.pro_descripcion', $search)
                ->orLike('p.pro_marca', $search)
                ->orLike('p.pro_familia', $search)
              ->groupEnd();
        }

        // 5) total
        $total = $b->countAllResults(false);

        // 6) orden & paginacin (por chunk)
        $b->orderBy('p.pro_codigo', 'asc');
        if ($len === -1) {
            $b->limit(50000);
        } else {
            $b->limit($len, $start);
        }

        $rows = $b->get()->getResultArray();

        // 7) catlogo de campos de la matriz
        $campos = $this->db->table('public.tbl_matriz_campo_ref mcr')
            ->select('mcr.*, cg.cam_nombre, cg.cam_origen, cg.cam_tipo, cg.cam_source, cg.cam_formula, cg.cam_depends, cg.cam_editable')
            ->join('public.tbl_campo_global cg','cg.cam_id = mcr.cam_id','inner')
            ->where('mcr.mat_id', (int)$matId)
            ->orderBy('mcr.mcr_orden','asc')
            ->get()->getResultArray();

        // 7.1) ALLOWLIST DIN09MICA: registra todas las fuentes declaradas en el catlogo
        $resolver = $this->resolver ?? new FieldResolver();
        foreach ($campos as $c) {
            $src = $c['cam_source'] ?? null;
            if (is_string($src) && $src !== '') $src = json_decode($src, true);
            if (is_array($src) && !empty($src['table']) && !empty($src['column'])) {
                $resolver->addAllowlistTable((string)$src['table'], [
                    (string)$src['column'], 'updated_at', 'pro_codigo', 'pro_reference_id'
                ]);
            }
        }

        // 8) resolver por fila
        $data = [];
        foreach ($rows as $r) {
            $line = [
                'pro_id'            => $r['pro_id'],
                'pro_codigo'        => $r['pro_codigo'],
                'pro_descripcion'   => $r['pro_descripcion'],
                'pro_familia'       => $r['pro_familia'],
                'pro_categorizacion'=> $r['pro_categorizacion'],
            ];

            // manuales guardados
            $mad = [];
            if (!empty($r['mad_campos'])) {
                $mad = is_array($r['mad_campos']) ? $r['mad_campos'] : (json_decode($r['mad_campos'], true) ?: []);
            }

            // contexto inicial con manuales/overrides
            $ctx = $mad;

            foreach ($campos as $c) {
                $name = $c['cam_nombre'];

                // Lectura sin source = manual (desde mad_campos)
                if (($c['cam_origen'] ?? 'lectura') === 'lectura' && empty($c['cam_source'])) {
                    $line[$name] = array_key_exists($name, $mad) ? $mad[$name] : null;
                    continue;
                }

                try {
                    $val = $resolver->resolveField($c, $r, null, $ctx);
                } catch (\Throwable $e) {
                    log_message('error', "resolveField {$name} -> ".$e->getMessage());
                    $val = null;
                }
                $line[$name] = $val;
            }

            $data[] = $line;
        }

        return $this->response->setJSON([
            'draw'            => $draw,
            'recordsTotal'    => $total,
            'recordsFiltered' => $total,
            'data'            => $data,
        ]);
    }

    // ===== API: guardar valor manual inline (editables de lectura/param)
    public function saveManual($matId)
    {
        $proId = (int)$this->request->getPost('pro_id');
        $name  = trim((string)$this->request->getPost('name'));
        $val   = $this->request->getPost('value');

        if (!$proId || $name==='') {
            return $this->response->setStatusCode(422)->setJSON(['ok'=>false,'msg'=>'pro_id y name requeridos']);
        }

        // Normaliza segn tipo (percent/number/money)
        $campo = $this->db->table('public.tbl_campo_global')->where('cam_nombre', $name)->get()->getRowArray();
        $tipo  = strtolower((string)($campo['cam_tipo'] ?? ''));

        if ($tipo === 'percent') {
            $raw = is_string($val) ? str_replace(['%',' ', ','], ['', '', '.'], (string)$val) : $val;
            $num = is_numeric($raw) ? (float)$raw : null;
            $val = ($num === null) ? null : ($num / 100.0); // guardar como fraccin
        } elseif ($tipo === 'number' || $tipo === 'money') {
            $raw = is_string($val) ? str_replace(',', '.', (string)$val) : $val;
            $val = is_numeric($raw) ? (float)$raw : null;
        }

        $ok = $this->mad->upsertCampos((int)$matId, $proId, null, [$name=>$val]);
        return $this->response->setJSON(['ok'=>$ok]);
    }

    // ===== API: reclculo en lote (placeholder para batch/cron)
    public function recalc($matId)
    {
        // Aqu puedes disparar un Job/CLI/cron que materialice resultados a mad_cache.
        return $this->response->setJSON(['ok'=>true,'msg'=>'Recalcular lanzado (implementar batch/cron segn tama09o).']);
    }

    // ===== API: evala una fila aplicando overrides (enter en el grid)
    public function eval($matId)
    {
        $proId     = (int)$this->request->getPost('pro_id');
        $overrides = json_decode((string)$this->request->getPost('overrides'), true) ?: [];

        if (!$proId) {
            return $this->response->setStatusCode(422)->setJSON(['ok'=>false,'msg'=>'pro_id requerido']);
        }

        // 1) fila base
        $r = $this->db->table('public.tbl_producto p')
            ->select('p.pro_id, p.pro_codigo, p.pro_descripcion, p.pro_familia, p.pro_categorizacion, p.pro_marca, p.pro_reference_id,
                      d.mad_id, d.mad_campos')
            ->join('public.tbl_matriz_dato d', 'd.pro_id = p.pro_id AND d.mat_id = '.(int)$matId, 'left')
            ->where('p.pro_id', $proId)
            ->get()->getRowArray();

        if (!$r) return $this->response->setStatusCode(404)->setJSON(['ok'=>false,'msg'=>'producto no encontrado']);

        // 2) mergea manuales + overrides
        $mad = [];
        if (!empty($r['mad_campos'])) {
            $mad = is_array($r['mad_campos']) ? $r['mad_campos'] : (json_decode($r['mad_campos'], true) ?: []);
        }
        $mad = array_merge($mad, $overrides);

        // 3) catlogo de campos
        $campos = $this->db->table('public.tbl_matriz_campo_ref mcr')
            ->select('mcr.*, cg.cam_nombre, cg.cam_origen, cg.cam_tipo, cg.cam_source, cg.cam_formula, cg.cam_depends')
            ->join('public.tbl_campo_global cg','cg.cam_id = mcr.cam_id','inner')
            ->where('mcr.mat_id', (int)$matId)
            ->orderBy('mcr.mcr_orden','asc')
            ->get()->getResultArray();

        // 3.1) ALLOWLIST DIN09MICA para todas las fuentes del catlogo
        $resolver = $this->resolver ?? new FieldResolver();
        foreach ($campos as $c) {
            $src = $c['cam_source'] ?? null;
            if (is_string($src) && $src !== '') $src = json_decode($src, true);
            if (is_array($src) && !empty($src['table']) && !empty($src['column'])) {
                $resolver->addAllowlistTable((string)$src['table'], [
                    (string)$src['column'], 'updated_at', 'pro_codigo', 'pro_reference_id'
                ]);
            }
        }

        // 4) resolver
        $ctx = $mad; // que las frmulas vean overrides
        $line = [
            'pro_id'            => $r['pro_id'],
            'pro_codigo'        => $r['pro_codigo'],
            'pro_descripcion'   => $r['pro_descripcion'],
            'pro_familia'       => $r['pro_familia'],
            'pro_categorizacion'=> $r['pro_categorizacion'],
        ];

        foreach ($campos as $c) {
            $name = $c['cam_nombre'];

            if (($c['cam_origen'] ?? 'lectura') === 'lectura' && empty($c['cam_source'])) {
                $line[$name] = array_key_exists($name, $mad) ? $mad[$name] : null;
                continue;
            }
            try {
                $val = $resolver->resolveField($c, $r, null, $ctx);
            } catch (\Throwable $e) {
                log_message('error', "eval resolveField {$name} -> ".$e->getMessage());
                $val = null;
            }
            $line[$name] = $val;
        }

        return $this->response->setJSON(['ok'=>true,'row'=>$line]);
    }
    public function debugSources($matId)
{
    @ini_set('max_execution_time', '180');
    @set_time_limit(180);

    $limit  = max(1, (int)($this->request->getGet('limit')  ?? 2000));   // cuntas filas muestrear
    $offset = max(0, (int)($this->request->getGet('offset') ?? 0));      // desplazamiento
    $withFormulas = (int)($this->request->getGet('formulas') ?? 1);      // 1 = evala frmulas

    // 1) Catlogo de la matriz
    $campos = $this->db->table('public.tbl_matriz_campo_ref mcr')
        ->select('mcr.*, cg.cam_nombre, cg.cam_origen, cg.cam_tipo, cg.cam_source, cg.cam_formula, cg.cam_depends')
        ->join('public.tbl_campo_global cg','cg.cam_id = mcr.cam_id','inner')
        ->where('mcr.mat_id', (int)$matId)
        ->orderBy('mcr.mcr_orden','asc')
        ->get()->getResultArray();

    if (!$campos) {
        return $this->response->setJSON(['ok'=>false,'msg'=>'Matriz sin campos']);
    }

    // 2) Productos (muestra)
    $q = $this->db->table('public.tbl_producto p')
        ->select('p.pro_id, p.pro_codigo, p.pro_descripcion, p.pro_familia, p.pro_categorizacion, p.pro_marca, p.pro_reference_id,
                  d.mad_id, d.mad_campos')
        ->join('public.tbl_matriz_dato d', 'd.pro_id = p.pro_id AND d.mat_id = '.(int)$matId, 'left')
        ->where("COALESCE(p.pro_activo,'true') = 'true'", null, false)
        ->orderBy('p.pro_codigo','asc')
        ->limit($limit, $offset);

    $rows = $q->get()->getResultArray();
    $totalScanned = count($rows);

    // 3) Resolver con allowlist dinmica (todas las fuentes del catlogo)
    $resolver = $this->resolver ?? new \App\Libraries\FieldResolver();
    foreach ($campos as $c) {
        $src = $c['cam_source'] ?? null;
        if (is_string($src) && $src !== '') $src = json_decode($src, true);
        if (is_array($src) && !empty($src['table']) && !empty($src['column'])) {
            $resolver->addAllowlistTable((string)$src['table'], [
                (string)$src['column'], 'updated_at', 'pro_codigo', 'pro_reference_id'
            ]);
        }
    }

    // 4) Prepara estructuras de conteo
    $report = []; // cam_nombre => stats
    foreach ($campos as $c) {
        $name = (string)($c['cam_nombre'] ?? '');
        if ($name === '') continue;
        $report[$name] = [
            'cam_nombre' => $name,
            'origen'     => (string)($c['cam_origen'] ?? ''),
            'tipo'       => (string)($c['cam_tipo'] ?? ''),
            'source'     => null,
            'nulls'      => 0,
            'non_nulls'  => 0,
            'samples'    => [],
            'errors'     => 0,
            'formula'    => [
                'depends'          => [],
                'depends_missing'  => [],
            ],
        ];

        // adjunta fuente si es lectura con source
        $src = $c['cam_source'] ?? null;
        if (is_string($src) && $src !== '') $src = json_decode($src, true);
        if (is_array($src) && !empty($src['table']) && !empty($src['column'])) {
            $report[$name]['source'] = [
                'table'  => (string)$src['table'],
                'column' => (string)$src['column'],
            ];
        }

        // dependencias (si frmula)
        if (strtolower((string)$c['cam_origen']) === 'formula') {
            $deps = $c['cam_depends'] ?? [];
            if (!is_array($deps)) $deps = json_decode((string)$deps, true) ?: [];
            $report[$name]['formula']['depends'] = array_values($deps);

            // detecta dependencias que no existen en catlogo
            $missing = [];
            foreach ($deps as $d) {
                $def = $this->db->table('public.tbl_campo_global')
                    ->select('cam_id')->where('cam_nombre', $d)->get()->getRowArray();
                if (!$def) $missing[] = $d;
            }
            $report[$name]['formula']['depends_missing'] = $missing;
        }
    }

    // 5) Itera filas y mide nulls/errores/muestras
    foreach ($rows as $r) {
        // manuales guardados por fila
        $mad = [];
        if (!empty($r['mad_campos'])) {
            $mad = is_array($r['mad_campos']) ? $r['mad_campos'] : (json_decode($r['mad_campos'], true) ?: []);
        }
        // contexto arranca con manuales/overrides
        $ctx = $mad;

        foreach ($campos as $c) {
            $name   = (string)$c['cam_nombre'];
            $origen = strtolower((string)($c['cam_origen'] ?? 'lectura'));

            try {
                $val = null;

                if ($origen === 'lectura') {
                    $src = $c['cam_source'] ?? null;
                    if (is_string($src) && $src !== '') $src = json_decode($src, true);
                    if (is_array($src) && !empty($src['table']) && !empty($src['column'])) {
                        // lectura directa desde source
                        $val = $resolver->readFromSource($src, $r, null);
                    } else {
                        // manual desde mad_campos/contexto
                        $val = array_key_exists($name, $ctx) ? $ctx[$name] : null;
                    }
                    // cachea en ctx por si otros dependen
                    $ctx[$name] = $val;
                } elseif ($origen === 'param') {
                    $val = $c['cam_formula'] ?? null;
                } elseif ($origen === 'formula' && $withFormulas) {
                    // asegrate de tener dependencias en ctx
                    $deps = $c['cam_depends'] ?? [];
                    if (!is_array($deps)) $deps = json_decode((string)$deps, true) ?: [];
                    foreach ($deps as $d) {
                        if (!array_key_exists($d, $ctx)) {
                            // resuelve dependiente (si existe en catlogo)
                            $depDef = $this->db->table('public.tbl_campo_global')
                                ->where('cam_nombre', $d)->get()->getRowArray();
                            if ($depDef) {
                                // utiliza el mismo resolver en cascada
                                $ctx[$d] = $this->resolver->resolveField($depDef, $r, null, $ctx);
                            } else {
                                $ctx[$d] = null;
                            }
                        }
                    }
                    $val = $this->resolver->resolveField($c, $r, null, $ctx);
                }

                if ($val === null || $val === '') $report[$name]['nulls']++;
                else {
                    $report[$name]['non_nulls']++;
                    if (count($report[$name]['samples']) < 3) {
                        $report[$name]['samples'][] = is_scalar($val) ? $val : json_encode($val);
                    }
                }
            } catch (\Throwable $e) {
                $report[$name]['errors']++;
            }
        }
    }

    return $this->response->setJSON([
        'ok'               => true,
        'mat_id'           => (int)$matId,
        'scanned_rows'     => $totalScanned,
        'limit'            => $limit,
        'offset'           => $offset,
        'evaluatedFormulas'=> (bool)$withFormulas,
        'fields'           => array_values($report),
    ]);
}


public function createVersion($matId)
{
    @ini_set('max_execution_time', '300');
    @set_time_limit(300);

    $p = $this->request->getPost();
    $tipo   = strtoupper(trim($p['tipo'] ?? 'PDV'));            // 'DIST' | 'PDV' | 'COSTOS'
    $hasta  = trim($p['hasta'] ?? '');                           // YYYY-MM-DD
    $promo  = filter_var(($p['promo'] ?? false), FILTER_VALIDATE_BOOL);
    $motivo = trim($p['motivo'] ?? '');
    $usuario= trim($p['usuario'] ?? ($this->request->getVar('usuario') ?? 'SYSTEM'));

    // 1) Arma nombre legible
    $nameParts = [$tipo, $hasta !== '' ? $hasta : date('Y-m-d'), $promo ? 'promo=S' : 'promo=No'];
    if ($motivo !== '') $nameParts[] = mb_strimwidth($motivo, 0, 40, '', 'UTF-8');
    $mver_nombre = implode('  ', $nameParts);

    // 2) Reutiliza la lgica de data() pero SIN paginar: todas las filas/columnas
    $resolver = $this->resolver ?? new \App\Libraries\FieldResolver();

    // Catlogo de campos de la matriz (en orden)
    $campos = $this->db->table('public.tbl_matriz_campo_ref mcr')
        ->select('mcr.*, cg.cam_nombre, cg.cam_origen, cg.cam_tipo, cg.cam_source, cg.cam_formula, cg.cam_depends')
        ->join('public.tbl_campo_global cg','cg.cam_id = mcr.cam_id','inner')
        ->where('mcr.mat_id', (int)$matId)
        ->orderBy('mcr.mcr_orden','asc')
        ->get()->getResultArray();

    // Allowlist dinmica
    foreach ($campos as $c) {
        $src = $c['cam_source'] ?? null;
        if (is_string($src) && $src !== '') $src = json_decode($src, true);
        if (is_array($src) && !empty($src['table']) && !empty($src['column'])) {
            $resolver->addAllowlistTable((string)$src['table'], [
                (string)$src['column'], 'updated_at', 'pro_codigo', 'pro_reference_id'
            ]);
        }
    }

    // Productos activos (iterar por chunks para memoria estable)
    $baseQ = $this->db->table('public.tbl_producto p')
        ->select('p.pro_id, p.pro_codigo, p.pro_descripcion, p.pro_familia, p.pro_categorizacion, p.pro_marca, p.pro_reference_id,
                  d.mad_id, d.mad_campos')
        ->join('public.tbl_matriz_dato d', 'd.pro_id = p.pro_id AND d.mat_id = '.(int)$matId, 'left')
        ->where("COALESCE(p.pro_activo,'true') = 'true'", null, false)
        ->orderBy('p.pro_codigo','asc');

    // Conteo total
    $total = (clone $baseQ)->countAllResults(false);

    $CHUNK = 5000;
    $offset= 0;
    $rowsAll = [];

    while (true) {
        $q = (clone $baseQ)->limit($CHUNK, $offset);
        $rows = $q->get()->getResultArray();
        if (!$rows) break;

        foreach ($rows as $r) {
            // base visibles
            $line = [
                'pro_id'            => $r['pro_id'],
                'pro_codigo'        => $r['pro_codigo'],
                'pro_descripcion'   => $r['pro_descripcion'],
                'pro_familia'       => $r['pro_familia'],
                'pro_categorizacion'=> $r['pro_categorizacion'],
            ];
            // manuales por fila
            $mad = [];
            if (!empty($r['mad_campos'])) {
                $mad = is_array($r['mad_campos']) ? $r['mad_campos'] : (json_decode($r['mad_campos'], true) ?: []);
            }
            $ctx = $mad;

            // resolver cada campo de catlogo
            foreach ($campos as $c) {
                $name   = $c['cam_nombre'];
                $origen = strtolower((string)($c['cam_origen'] ?? 'lectura'));
                try {
                    if ($origen === 'lectura') {
                        $src = $c['cam_source'] ?? null;
                        if (is_string($src) && $src !== '') $src = json_decode($src, true);
                        if (is_array($src) && !empty($src['table']) && !empty($src['column'])) {
                            $val = $resolver->readFromSource($src, $r, null);
                        } else {
                            $val = array_key_exists($name, $ctx) ? $ctx[$name] : null;
                        }
                        $ctx[$name] = $val;
                    } elseif ($origen === 'param') {
                        $val = $c['cam_formula'] ?? null;
                    } else { // frmula
                        $val = $resolver->resolveField($c, $r, null, $ctx);
                    }
                } catch (\Throwable $e) {
                    log_message('error', "snapshot resolve {$name} -> ".$e->getMessage());
                    $val = null;
                }
                $line[$name] = $val;
            }
            $rowsAll[] = $line;
        }

        $offset += count($rows);
        if ($offset >= $total) break;
    }

    // 3) Arma registro
    $meta = [
        'hasta'  => $hasta,
        'promo'  => $promo,
        'motivo' => $motivo,
    ];

    $ver = new MatrizVersionModel();
    $mver_id = $ver->insert([
        'mat_id'       => (int)$matId,
        'mver_tipo'    => in_array($tipo, ['DIST','PDV','COSTOS']) ? $tipo : 'PDV',
        'mver_nombre'  => $mver_nombre,
        'mver_usuario' => $usuario,
        'mver_meta'    => json_encode($meta, JSON_UNESCAPED_UNICODE),
        'mver_rows'    => count($rowsAll),
        'payload'      => json_encode($rowsAll, JSON_UNESCAPED_UNICODE),
        'created_at'   => date('c'),
    ], true);

    // 4) Retencin: borra >180 das (una barrida ligera)
    $this->db->query("DELETE FROM public.tbl_matriz_version WHERE created_at < now() - interval '180 days'");

    return $this->response->setJSON(['ok'=>true,'mver_id'=>$mver_id, 'rows'=>count($rowsAll)]);
}
public function versions($matId)
{
    $q      = trim((string)$this->request->getGet('q') ?? '');
    $tipo   = strtoupper(trim((string)$this->request->getGet('tipo') ?? '')); // DIST|PDV|COSTOS|''

    $w = $this->db->table('public.tbl_matriz_version')
        ->select('mver_id, mver_tipo, mver_nombre, mver_usuario, mver_rows, created_at, mver_meta')
        ->where('mat_id', (int)$matId);

    if ($tipo !== '' && in_array($tipo, ['DIST','PDV','COSTOS'])) $w->where('mver_tipo', $tipo);
    if ($q !== '') $w->like('mver_nombre', $q);

    $w->orderBy('created_at','desc');

    $data = $w->get()->getResultArray();

    // vista simple con filtros + exportar JSON
    echo view('layouts/header');
    echo view('layouts/aside');
    echo view('matrices2/versions', ['mat_id'=>$matId, 'items'=>$data]);
    echo view('layouts/footer');
}

public function versionJson($mverId)
{
    $v = $this->db->table('public.tbl_matriz_version')
        ->where('mver_id', (int)$mverId)->get()->getRowArray();
    if (!$v) return $this->response->setStatusCode(404)->setJSON(['ok'=>false]);

    // descarga JSON
    return $this->response
        ->setHeader('Content-Type','application/json; charset=utf-8')
        ->setHeader('Content-Disposition','attachment; filename="matriz_version_'.$v['mver_id'].'.json"')
        ->setJSON(json_decode($v['payload'], true));
}
// En Matrices2Controller

public function versionView($mverId)
{
    $v = $this->db->table('public.tbl_matriz_version')->where('mver_id', (int)$mverId)->get()->getRowArray();
    if (!$v) return redirect()->to('/matrices2');

    // Decodifica payload (matriz completa) + meta para encabezado
    $rows = json_decode($v['payload'] ?? '[]', true) ?: [];
    $meta = json_decode($v['mver_meta'] ?? '{}', true) ?: [];

    // Derivar columnas desde la primera fila
    $columns = [];
    $headers = [];
    if (!empty($rows)) {
        $first = $rows[0];
        foreach ($first as $k => $val) {
            $columns[] = ['data' => $k, 'readOnly' => true, 'width' => 120];
            // Etiquetas ms amigables en base a clave
            $title = $k;
            if     ($k === 'pro_codigo')         $title = 'Cdigo';
            elseif ($k === 'pro_descripcion')    $title = 'Descripcin';
            elseif ($k === 'pro_familia')        $title = 'Familia';
            elseif ($k === 'pro_categorizacion') $title = 'Categorizacin';
            $headers[] = $title;
        }
    }

    echo view('layouts/header');
    echo view('layouts/aside');
    echo view('matrices2/version_view', [
        'version' => $v,
        'rows'    => $rows,
        'columns' => $columns,
        'headers' => $headers,
        'meta'    => $meta,
    ]);
    echo view('layouts/footer');
}

}
