<?php namespace App\Controllers;

use App\Models\ProductoModel;
use App\Models\ParametrosProductoModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ProductosImportController extends Controller
{
    public function index()
    {
        echo view('layouts/header');
        echo view('layouts/aside');
        echo view('productos/importar_matriz'); // la vista de abajo
        echo view('layouts/footer');
    }

    public function plantilla()
    {
        $headers = [
            'pro_codigo','ppr_margen_hoy',
            'ppr_desc_normal','ppr_sentido_normal',
            'ppr_desc_limite','ppr_sentido_limite',
            'ppr_desc_admin','ppr_sentido_admin',
            'ppr_desc_mayorista','ppr_sentido_mayorista',
            'ppr_desc_especial','ppr_sentido_especial',
            'ppr_observacion','ppr_activo'
        ];
    
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->fromArray($headers, null, 'A1');
        $sheet->getStyle('A1:N1')->getFont()->setBold(true);
        // Fila ejemplo
        $sheet->fromArray([[
            'ABC123','0.3500',
            '5.0000','D',
            '0.0000','D',
            '0.0000','D',
            '10.0000','D',
            '15.0000','D',
            'Carga masiva','1'
        ]], null, 'A2');
    
        $writer = \PhpOffice\PhpSpreadsheet\IOFactory::createWriter($spreadsheet, 'Xlsx');
    
        // Escribe a un archivo temporal y descárgalo
        $tmp = tempnam(sys_get_temp_dir(), 'tpl_').'.xlsx';
        $writer->save($tmp);
    
        return $this->response
            ->download($tmp, null)
            ->setFileName('plantilla_parametros_producto.xlsx')
            ->setContentType('application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    }


    public function preview()
    {
        helper(['filesystem']);
        $file = $this->request->getFile('archivo');
        if (!$file || !$file->isValid()) {
            return redirect()->back()->with('error','Archivo inválido.');
        }
        if (strtolower($file->getExtension()) !== 'xlsx') {
            return redirect()->back()->with('error','Solo se acepta Excel (.xlsx).');
        }

        // Guardar en writable/uploads con token
        $token = bin2hex(random_bytes(8));
        $dest  = WRITEPATH.'uploads/param_import_'.$token.'.xlsx';
        $file->move(dirname($dest), basename($dest));

        // Parsear para preview (podemos mostrar todas, o limitar render a 1000 pero contar todas)
        $reader = IOFactory::createReader('Xlsx');
        $reader->setReadDataOnly(true);
        $spreadsheet = $reader->load($dest);
        $sheet = $spreadsheet->getActiveSheet();

        $rowsPreview = [];
        $validCount = 0; $invalidCount = 0; $total = 0;

        $productoModel = new ProductoModel();

        // Lee todas las filas
        $highestRow = $sheet->getHighestRow();
        $highestCol = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($sheet->getHighestColumn());

        // Mapa encabezados
        $headers = [];
        for ($c=1; $c<=$highestCol; $c++){
            $headers[$c] = trim((string)$sheet->getCellByColumnAndRow($c, 1)->getValue());
        }
        $required = ['pro_codigo','ppr_margen_hoy','ppr_desc_normal','ppr_sentido_normal'];
        foreach ($required as $req) {
            if (!in_array($req, $headers, true)) {
                return redirect()->back()->with('error','La plantilla no contiene la columna requerida: '.$req);
            }
        }

        // Índices
        $idx = array_flip($headers);

        for ($r=2; $r<=$highestRow; $r++) {
            $total++;
            $get = function($name) use ($sheet,$r,$idx){
                if (!isset($idx[$name])) return null;
                return trim((string)$sheet->getCellByColumnAndRow($idx[$name], $r)->getValue());
            };

            $pro = $get('pro_codigo');
            if ($pro==='') { $invalidCount++; continue; }

            $existe = $productoModel->existeCodigo($pro);
            $mho = (float) str_replace(',', '.', ($get('ppr_margen_hoy') ?? 0));
            $dn  = (float) str_replace(',', '.', ($get('ppr_desc_normal') ?? 0));
            $dl  = (float) str_replace(',', '.', ($get('ppr_desc_limite') ?? 0));
            $da  = (float) str_replace(',', '.', ($get('ppr_desc_admin') ?? 0));
            $dy  = (float) str_replace(',', '.', ($get('ppr_desc_mayorista') ?? 0));
            $de  = (float) str_replace(',', '.', ($get('ppr_desc_especial') ?? 0));

            $rowPreview = [
                'pro_codigo' => $pro,
                'ppr_margen_hoy' => number_format($mho,4,'.',''),
                'p_normal' => number_format($dn,4,'.',''),
                'p_limite' => number_format($dl,4,'.',''),
                'p_admin'  => number_format($da,4,'.',''),
                'p_mayor'  => number_format($dy,4,'.',''),
                'p_esp'    => number_format($de,4,'.',''),
                'valido'   => $existe && $mho>=0
            ];

            if ($rowPreview['valido']) $validCount++; else $invalidCount++;

            // Para DataTable de preview mostramos hasta 1000 filas para no reventar el DOM
            if (count($rowsPreview) < 1000) $rowsPreview[] = $rowPreview;
        }
        echo view('layouts/header');
        echo view('layouts/aside');
        echo view('productos/importar_matriz', [
            'token' => $token,
            'previewRows' => $rowsPreview,
            'total' => $total,
            'validos' => $validCount,
            'invalidos' => $invalidCount,
        ]); // la vista de abajo
        echo view('layouts/footer');
        
    }

    public function apply()
    {
        $token = $this->request->getPost('token');
        if (!$token) return redirect()->back()->with('error','Token no encontrado.');

        $path = WRITEPATH.'uploads/param_import_'.$token.'.xlsx';
        if (!is_file($path)) return redirect()->to(base_url('productos/importar'))->with('error','Archivo no disponible.');

        set_time_limit(0);

        $productoModel   = new ProductoModel();
        $paramModel      = new ParametrosProductoModel();

        $reader = IOFactory::createReader('Xlsx');
        $reader->setReadDataOnly(true);
        $spreadsheet = $reader->load($path);
        $sheet = $spreadsheet->getActiveSheet();
        $highestRow = $sheet->getHighestRow();
        $highestCol = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($sheet->getHighestColumn());

        // headers
        $headers = [];
        for ($c=1; $c<=$highestCol; $c++){
            $headers[$c] = trim((string)$sheet->getCellByColumnAndRow($c, 1)->getValue());
        }
        $idx = array_flip($headers);

        $ok = 0; $fail = 0;

        // Procesamos en transacciones por lotes para 50k+
        $batchSize = 1000;
        $batchCount = 0;
        $this->db = \Config\Database::connect();
        $this->db->transBegin();

        try {
            for ($r=2; $r<=$highestRow; $r++) {
                $get = function($name) use ($sheet,$r,$idx){
                    if (!isset($idx[$name])) return null;
                    return trim((string)$sheet->getCellByColumnAndRow($idx[$name], $r)->getValue());
                };

                $code = $get('pro_codigo');
                if ($code==='') { $fail++; continue; }

                if (!$productoModel->existeCodigo($code)) { $fail++; continue; }

                $mho = (float) str_replace(',', '.', ($get('ppr_margen_hoy') ?? 0));
                $dn  = (float) str_replace(',', '.', ($get('ppr_desc_normal') ?? 0));
                $dl  = (float) str_replace(',', '.', ($get('ppr_desc_limite') ?? 0));
                $da  = (float) str_replace(',', '.', ($get('ppr_desc_admin') ?? 0));
                $dy  = (float) str_replace(',', '.', ($get('ppr_desc_mayorista') ?? 0));
                $de  = (float) str_replace(',', '.', ($get('ppr_desc_especial') ?? 0));

                $porcs = [
                    'normal'=>$dn,'limite'=>$dl,'admin'=>$da,'mayorista'=>$dy,'especial'=>$de
                ];
                $margenes = ParametrosProductoModel::calcularMargenes($mho, $porcs);

                $row = [
                    'pro_codigo'             => $code,
                    'ppr_margen_hoy'         => number_format($mho,4,'.',''),
                    'ppr_desc_normal'        => number_format($dn,4,'.',''),
                    'ppr_sentido_normal'     => $get('ppr_sentido_normal') ?: 'D',
                    'ppr_margen_normal'      => $margenes['normal'],

                    'ppr_desc_limite'        => number_format($dl,4,'.',''),
                    'ppr_sentido_limite'     => $get('ppr_sentido_limite') ?: 'D',
                    'ppr_margen_limite'      => $margenes['limite'],

                    'ppr_desc_admin'         => number_format($da,4,'.',''),
                    'ppr_sentido_admin'      => $get('ppr_sentido_admin') ?: 'D',
                    'ppr_margen_admin'       => $margenes['admin'],

                    'ppr_desc_mayorista'     => number_format($dy,4,'.',''),
                    'ppr_sentido_mayorista'  => $get('ppr_sentido_mayorista') ?: 'D',
                    'ppr_margen_mayorista'   => $margenes['mayorista'],

                    'ppr_desc_especial'      => number_format($de,4,'.',''),
                    'ppr_sentido_especial'   => $get('ppr_sentido_especial') ?: 'D',
                    'ppr_margen_especial'    => $margenes['especial'],

                    'ppr_observacion'        => $get('ppr_observacion') ?: null,
                    'ppr_activo'             => filter_var($get('ppr_activo'), FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) ?? true,
                    'ppr_created_at'         => date('c'),
                    'ppr_updated_at'         => date('c')
                ];

                try {
                    // Inactivar activo previo de ese producto
                    $paramModel->inactivarActivoPorCodigo($code);
                    // Insertar nuevo set activo
                    $paramModel->insert($row);
                    $ok++;
                } catch (\Throwable $e) {
                    $fail++;
                }

                // Commit en lotes para no bloquear mucho tiempo
                $batchCount++;
                if ($batchCount >= $batchSize) {
                    $this->db->transCommit();
                    $this->db->transBegin();
                    $batchCount = 0;
                }
            }

            $this->db->transCommit();
        } catch (\Throwable $e) {
            $this->db->transRollback();
            return redirect()->back()->with('error','Error al aplicar: '.$e->getMessage());
        }

        // Limpieza opcional del archivo
        @unlink($path);

        return redirect()->to(base_url('productos/importar'))
            ->with('ok', "Proceso completado. Válidos: $ok — Fallidos: $fail");
    }
}
