<?php
namespace App\Controllers;

use App\Models\PdvModel;
use App\Models\PdvPrecioHistModel;
use App\Models\PdvImportLogModel;

class SincronizacionPrecios extends BaseController
{
    private string $baseUri;
    private string $authRaw;
    private int    $pageSize   = 100; // la API permite 100
    private int    $maxRetries = 2;   // reintentos por página
    private int    $timeout    = 60;  // segundos

    public function __construct()
    {
        $this->baseUri = rtrim(getenv('precios.base_uri') ?: 'https://int.api.megaprofer.com/', '/') . '/';
        $this->authRaw = getenv('precios.auth_raw') ?: '';
        $this->timeout = (int) (getenv('precios.timeout') ?: $this->timeout);
    }

    /**
     * GET /sincronizar/precios-pdv?recId=65&page=0&size=100
     * GET /sincronizar/precios-pdv?recId=65&all=1&size=100  (recorre todas las páginas hasta vacío)
     */
    public function precios_pdv()
    {
        @set_time_limit(0);
        @ini_set('memory_limit', '1024M');

        $recId = (int)($this->request->getGet('recId') ?? 65);
        if ($recId <= 0) {
            return $this->response->setStatusCode(400)->setJSON(['ok'=>false,'error'=>'Parámetro recId requerido']);
        }

        $page = (int)($this->request->getGet('page') ?? 0);
        $size = (int)($this->request->getGet('size') ?? $this->pageSize);
        if ($size <= 0) $size = $this->pageSize;
        $all  = (int)($this->request->getGet('all') ?? 0);

        $mLog  = new PdvImportLogModel();
        $mPDV  = new PdvModel();
        $mHist = new PdvPrecioHistModel();

        $logId = $mLog->startRun($recId, $size);

        $ins=0; $ign=0; $pages=0; $errs=0; $totalElems=0;

        try {
            if ($all === 1) {
                $p=0;
                while (true) {
                    try {
                        $arr   = $this->fetchPageWithRetry($recId, $p, $size);
                        $batch = $this->normalizeItems($arr); // sin tocar listas
                        if (empty($batch['items'])) break;

                        if (!empty($batch['pdvName'])) {
                            $mPDV->upsertOne((string)$recId, $batch['pdvName']);
                        }

                        // histórico por lote (ON CONFLICT DO NOTHING por diff_hash)
                        $res = $mHist->insertBatchIfChanged($recId, $batch['items']);
                        $ins += (int)($res['inserted'] ?? 0);
                        $ign += (int)($res['ignored'] ?? 0);

                        $totalElems = max($totalElems, (int)($arr['totalElements'] ?? 0));
                        $pages++; $p++;
                    } catch (\Throwable $ePage) {
                        $errs++;
                        log_message('error', 'Sync precios PDV recId='.$recId.' page='.$p.' error: '.$ePage->getMessage());
                        $p++; // avanza para no quedar atascado
                    }
                }
            } else {
                $arr   = $this->fetchPageWithRetry($recId, $page, $size);
                $batch = $this->normalizeItems($arr);

                if (!empty($batch['pdvName'])) {
                    $mPDV->upsertOne((string)$recId, $batch['pdvName']);
                }

                $res = $mHist->insertBatchIfChanged($recId, $batch['items']);
                $ins += (int)($res['inserted'] ?? 0);
                $ign += (int)($res['ignored'] ?? 0);

                $totalElems = (int)($arr['totalElements'] ?? 0);
                $pages = 1;
            }

            // cerrar log
            $mLog->finishRun($logId, [
                'total_elements' => $totalElems,
                'rows_inserted'  => $ins,
                'rows_ignored'   => $ign,
                'pages'          => $pages,
                'errors'         => $errs,
            ]);

            return $this->response->setJSON([
                'ok' => $errs===0,
                'recId'=>$recId, 'pages'=>$pages, 'totalElements'=>$totalElems,
                'inserted'=>$ins, 'ignored'=>$ign, 'errors'=>$errs
            ]);

        } catch (\Throwable $e) {
            $mLog->finishRun($logId, [
                'status' => 'ERROR',
                'message'=> $e->getMessage()
            ]);
            return $this->response->setStatusCode(500)->setJSON(['ok'=>false,'error'=>$e->getMessage()]);
        }
    }

    /* ====================== HTTP helpers ====================== */

    private function fetchPageWithRetry(int $recId, int $page, int $size): array
    {
        $attempt = 0;
        while (true) {
            try {
                return $this->fetchPage($recId, $page, $size);
            } catch (\Throwable $e) {
                $attempt++;
                if ($attempt > $this->maxRetries) { throw $e; }
                sleep($attempt === 1 ? 2 : 5);
            }
        }
    }

    private function fetchPage(int $recId, int $page, int $size): array
    {
        $endpoint = $this->baseUri . 'pricing/price-product-pdv';
        $query = http_build_query([
            'recId' => $recId,
            'page'  => $page,
            'size'  => $size,
        ], '', '&', PHP_QUERY_RFC3986);

        $headers = [
            'Accept: application/json',
            'Content-Type: application/json',
            'Authorization: ' . $this->authRaw,
            'User-Agent: PreciosPDVSync/1.0',
        ];

        $body = $this->curlGet($endpoint.'?'.$query, $headers, $this->timeout);
        $arr  = json_decode($body, true);
        if (!is_array($arr)) {
            throw new \RuntimeException('Respuesta inválida del API (PDV) page='.$page);
        }
        return $arr;
    }

    private function curlGet(string $url, array $headers, int $timeout = 60): string
    {
        $ch = curl_init();
        if ($ch === false) throw new \RuntimeException('No se pudo inicializar cURL');

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_CONNECTTIMEOUT => $timeout,
            CURLOPT_TIMEOUT        => $timeout,
            CURLOPT_HTTPHEADER     => $headers,
        ]);
        $body = curl_exec($ch);
        if ($body === false) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new \RuntimeException('Error cURL: ' . $err);
        }
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($code < 200 || $code >= 300) {
            throw new \RuntimeException("HTTP $code: $body");
        }
        return $body;
    }

    /* ====================== Mapper ====================== */

    private function n4($v) { return $v === null ? null : round((float)$v, 4); }
    private function b($v)  { return is_bool($v) ? $v : (in_array($v, [1,'1','true',true], true)); }

    /**
     * Normaliza y prepara payload para histórico.
     * - NO consulta ni inserta listas
     * - Guarda solo lpr_rec_id (numérico) y price_list_name denormalizado
     */
    private function normalizeItems($apiResp)
    {
        $data = isset($apiResp['data']) ? $apiResp['data'] : [];

        // La API a veces responde con data = [ [ {...}, {...} ] ] → desanidar.
        if (is_array($data) && $this->isListArray($data)) {
            if (isset($data[0]) && is_array($data[0]) && $this->isListArray($data[0])) {
                $data = $data[0];
            }
        }

        $items = [];
        $pdvName = null;
        $priceListRecId = null;
        $priceListName  = null;

        foreach ($data as $row) {
            if (!is_array($row)) continue;

            if ($pdvName === null)        $pdvName        = (string)($row['establishmentName'] ?? '');
            if ($priceListRecId === null) $priceListRecId = isset($row['priceListRecId']) ? (int)$row['priceListRecId'] : null;
            if ($priceListName === null)  $priceListName  = (string)($row['priceListName'] ?? '');

            // clave de producto
            $proCodigo = (string)($row['productReferenceCode'] ?? '');
            if ($proCodigo === '') continue;

            $payload = [
                'lpr_rec_id'        => $priceListRecId,  // SOLO FK numérica
                'price_list_name'   => $priceListName,   // denormalizado
                'pro_codigo'        => $proCodigo,
                'product_uuid'      => $row['productuuid'] ?? null,
                'measury_reference_id' => (string)($row['measuryReferenceId'] ?? ''),
                'product_name'      => (string)($row['productName'] ?? ''),
                'measury_unit'      => (string)($row['measuryUnit'] ?? ''),
                'measury_unit_base' => $this->b($row['measuryUnitBase'] ?? null),
                'is_heavy'          => $this->b($row['isHeavy'] ?? null),
                'mark'              => (string)($row['mark'] ?? ''),
                'is_approved'       => $this->b($row['isApproved'] ?? null),
                'is_modified'       => $this->b($row['isModified'] ?? null),
                'is_hook'           => $this->b($row['isHook'] ?? null),
                'margin_price_pdv'  => isset($row['marginPricePdv']) && $row['marginPricePdv']!==null ? round((float)$row['marginPricePdv'], 2) : null,

                'price'               => $this->n4($row['price'] ?? null),
                'price_limit'         => $this->n4($row['priceLimit'] ?? null),
                'normal_price'        => $this->n4($row['normalPrice'] ?? null),
                'wholesaler_price'    => $this->n4($row['wholeSalerPrice'] ?? null),
                'special_price'       => $this->n4($row['specialPrice'] ?? null),
                'administrator_price' => $this->n4($row['administratorPrice'] ?? null),
                'base_price'          => $this->n4($row['basePrice'] ?? null),

                'updated_api_at'    => !empty($row['updated']) ? date('c', strtotime($row['updated'])) : null,
                'raw_json'          => json_encode($row, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
            ];

            // Hash de cambio material (incluye la lista numérica)
            $hashBase = [
                $payload['product_uuid'],
                $payload['product_name'],
                $payload['measury_unit'],
                $payload['measury_unit_base'],
                $payload['is_heavy'],
                $payload['mark'],
                $payload['is_approved'],
                $payload['is_modified'],
                $payload['is_hook'],
                $payload['margin_price_pdv'],
                $payload['price'],
                $payload['price_limit'],
                $payload['normal_price'],
                $payload['wholesaler_price'],
                $payload['special_price'],
                $payload['administrator_price'],
                $payload['base_price'],
                $priceListRecId,        // <- solo recId para diferenciar por lista
            ];
            $payload['diff_hash'] = md5(json_encode($hashBase));

            $items[] = $payload;
        }

        return [
            'pdvName'        => $pdvName,
            'priceListRecId' => $priceListRecId, // informativo (no se usa ya para upsert listas)
            'priceListName'  => $priceListName,  // informativo
            'items'          => $items,
        ];
    }

    private function isListArray(array $a): bool
    {
        return array_keys($a) === range(0, count($a) - 1);
    }
}
