<?php
namespace App\Models;

use CodeIgniter\Model;

class ListaPreciosModel extends Model
{
    protected $table         = 'tbl_lista_precios';
    protected $primaryKey    = 'lpr_id';
    protected $allowedFields = [
        'lpr_rec_id', 'lpr_nombre',
        'lpr_ref_id', 'ext_id', 'active',
        'created_at', 'updated_at'
    ];
    protected $useTimestamps = true;
    protected $createdField  = 'created_at';
    protected $updatedField  = 'updated_at';
    protected $returnType    = 'array';

    /** Upsert masivo desde /pricing/price-lists (clave lpr_ref_id) */
    public function upsertManyFromApi(array $items): array
{
    $db = $this->db;
    $inserted = 0; 
    $updated  = 0;

    foreach ($items as $it) {
        $refId = isset($it['refId']) ? (string)$it['refId'] : '';
        if ($refId === '') continue;

        $name   = (string)($it['name'] ?? '');
        $extId  = (string)($it['extId'] ?? '');
        $active = array_key_exists('active', $it) ? $it['active'] : null;

        // Si el referenceId luce numrico, lo usamos como recId; si no, NULL
        $recId  = null;
        if (is_numeric($refId)) {
            $recId = (int)$refId;
        }

        // 1) UPDATE por lpr_ref_id (si ya existe esa clave lgica)
        $res1 = $db->query("
            UPDATE public.tbl_lista_precios
               SET lpr_nombre = :name:,
                   ext_id     = :extId:,
                   active     = :active:,
                   lpr_rec_id = COALESCE(:recId:, lpr_rec_id),
                   updated_at = now()
             WHERE lpr_ref_id = :refId:
        ", [
            'name'=>$name,'extId'=>$extId,'active'=>$active,'recId'=>$recId,'refId'=>$refId
        ]);

        if ($db->affectedRows() > 0) { $updated++; continue; }

        // 2) UPDATE por lpr_rec_id (cuando ya existe la fila antigua por recId)
        if ($recId !== null) {
            $res2 = $db->query("
                UPDATE public.tbl_lista_precios
                   SET lpr_ref_id = :refId:,
                       lpr_nombre = :name:,
                       ext_id     = :extId:,
                       active     = :active:,
                       updated_at = now()
                 WHERE lpr_rec_id = :recId:
            ", [
                'refId'=>$refId,'name'=>$name,'extId'=>$extId,'active'=>$active,'recId'=>$recId
            ]);
            if ($db->affectedRows() > 0) { $updated++; continue; }
        }

        // 3) INSERT (no existe ni por refId ni por recId)
        $db->query("
            INSERT INTO public.tbl_lista_precios
                (lpr_ref_id, lpr_nombre, ext_id, active, lpr_rec_id, created_at, updated_at)
            VALUES
                (:refId:, :name:, :extId:, :active:, :recId:, now(), now())
        ", [
            'refId'=>$refId,'name'=>$name,'extId'=>$extId,'active'=>$active,'recId'=>$recId
        ]);
        $inserted++;
    }

    return ['inserted'=>$inserted, 'updated'=>$updated];
}


    /** Pruning: marca inactive las listas que no llegaron en esta corrida */
    public function pruneMissing(array $refIds): int
    {
        if (empty($refIds)) return 0;
        // usamos array literal de PG
        $arr = '{'.implode(',', array_map(fn($s)=>'"'.str_replace('"','\"',$s).'"',$refIds)).'}';
        $sql = "
          UPDATE public.tbl_lista_precios
             SET active = FALSE, updated_at = now()
           WHERE lpr_ref_id IS NOT NULL
             AND lpr_ref_id <> ALL(:arr:)
             AND active IS DISTINCT FROM FALSE
        ";
        $this->db->query($sql, ['arr'=>$arr]);
        return $this->db->affectedRows();
    }

    /** Helper: obtener lpr_ref_id por lpr_rec_id (para histórico de precios) */
    public function getRefIdByRecId(?int $recId): ?string
    {
        if (!$recId) return null;
        $q = $this->db->query(
            "SELECT lpr_ref_id FROM public.tbl_lista_precios WHERE lpr_rec_id = :rec: LIMIT 1",
            ['rec'=>$recId]
        );
        $val = $q->getRow('lpr_ref_id');
        return $val ?: null;
    }

    /** Helper: garantiza existencia por recId+name (cuando el histórico no encuentra la lista) */
    public function ensureByRecIdName(int $recId, string $name): void
    {
        $sql = "
          INSERT INTO public.tbl_lista_precios (lpr_rec_id, lpr_nombre, created_at, updated_at)
          VALUES (:rec:, :name:, now(), now())
          ON CONFLICT (lpr_rec_id) DO UPDATE
            SET lpr_nombre = COALESCE(EXCLUDED.lpr_nombre, public.tbl_lista_precios.lpr_nombre),
                updated_at = now()
        ";
        $this->db->query($sql, ['rec'=>$recId, 'name'=>$name ?: '']);
    }
}
