<?php namespace App\Services;

use Config\Services;
use Throwable;

/**
 * Servicio HTTP para la API de Precios (Megaprofer).
 * Compatible con CI4 + PHP 7.4.
 *
 * .env esperado:
 *   precios.base_uri = https://int.api.megaprofer.com/
 *   precios.timeout  = 60
 *   precios.auth_type = raw          ; none|raw|bearer|basic
 *   precios.auth_raw  = rapidito-int;N4RaVsn|&BaXLTIk]1
 *   precios.auth_token =
 *   precios.basic_user =
 *   precios.basic_pass =
 */
class PrecioApiService
{
    /** @var \CodeIgniter\HTTP\CURLRequest */
    protected $http;

    /** @var string */
    protected $base;

    /** @var int */
    protected $timeout;

    /** @var string none|raw|bearer|basic */
    protected $authType;

    /** @var string */
    protected $authRaw;

    /** @var string */
    protected $authToken;

    /** @var string */
    protected $basicUser;

    /** @var string */
    protected $basicPass;

    public function __construct()
    {
        $this->base      = rtrim(getenv('precios.base_uri') ?: 'https://int.api.megaprofer.com/', '/') . '/';
        $this->timeout   = (int) (getenv('precios.timeout') ?: 60);
        $this->authType  = (string) (getenv('precios.auth_type') ?: 'none');
        $this->authRaw   = (string) (getenv('precios.auth_raw') ?: '');
        $this->authToken = (string) (getenv('precios.auth_token') ?: '');
        $this->basicUser = (string) (getenv('precios.basic_user') ?: '');
        $this->basicPass = (string) (getenv('precios.basic_pass') ?: '');

        $this->http = Services::curlrequest([
            'baseURI'     => $this->base,
            'timeout'     => $this->timeout,
            'http_errors' => false,   // no lanzar excepción por status != 2xx
            'verify'      => true,    // SSL verificado
        ]);
    }

    /**
     * Cabeceras comunes + Authorization según .env
     */
    protected function headers(array $extra = []): array
    {
        $h = [
            'Accept'       => 'application/json',
            'Content-Type' => 'application/json',
        ];

        switch ($this->authType) {
            case 'raw':
                if ($this->authRaw !== '') {
                    $h['Authorization'] = $this->authRaw; // tal cual nos dieron
                }
                break;
            case 'bearer':
                if ($this->authToken !== '') {
                    $h['Authorization'] = 'Bearer ' . $this->authToken;
                }
                break;
            case 'basic':
                if ($this->basicUser !== '') {
                    $h['Authorization'] = 'Basic ' . base64_encode($this->basicUser . ':' . $this->basicPass);
                }
                break;
            case 'none':
            default:
                // sin auth
                break;
        }

        // Permite sobreescribir/añadir
        return array_merge($h, $extra);
    }

    /**
     * Envoltura de request con manejo simple de errores y decodificación JSON.
     * @param string $method GET|POST|PUT|DELETE
     * @param string $uri    ruta relativa (sin base)
     * @param array  $options ['headers'=>[],'query'=>[],'json'=>[]]
     * @param int    $retries reintentos (ej. 0,1,2)
     * @return array respuesta JSON decodificada (o ['_status'=>, '_body'=>] si no es JSON)
     * @throws \RuntimeException en fallo de transporte o demasiados intentos
     */
    protected function request(string $method, string $uri, array $options = [], int $retries = 0): array
    {
        // fusiona headers
        $options['headers'] = $this->headers($options['headers'] ?? []);

        $attempt = 0;
        do {
            try {
                $res = $this->http->request($method, ltrim($uri, '/'), $options);
                $status = $res->getStatusCode();
                $body   = (string) $res->getBody();

                // intenta JSON
                $json = json_decode($body, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($json)) {
                    // adjunta status por si quieres inspeccionarlo
                    $json['_status'] = $status;
                    return $json;
                }

                // si no es JSON, regresa crudo con status
                return [
                    '_status' => $status,
                    '_body'   => $body,
                ];
            } catch (Throwable $e) {
                $attempt++;
                if ($attempt > $retries) {
                    throw new \RuntimeException('Error al consumir API precios: ' . $e->getMessage(), 0, $e);
                }
                // backoff simple: 2s luego 5s...
                sleep($attempt === 1 ? 2 : 5);
            }
        } while ($attempt <= $retries);

        // no debería llegar aquí
        return ['_status' => 0, '_body' => ''];
    }

    // ============================================================
    // pricing-controller
    // ============================================================

    /**
     * POST /pricing/update-price-list
     * Actualiza el estado/atributos de una sublista.
     * $payload ejemplo:
     *  ["id"=>"61c...","referenceId"=>"1000057","name"=>"SUB LISTA..","active"=>true]
     */
    public function updatePriceList(array $payload, int $retries = 0): array
    {
        return $this->request('POST', 'pricing/update-price-list', [
            'json' => $payload,
        ], $retries);
    }

    /**
     * POST /pricing/prices
     * Envío/carga de precios.
     * $payload: estructura que defina tu plantilla Megapos.
     */
    public function postPrices(array $payload, int $retries = 0): array
    {
        return $this->request('POST', 'pricing/prices', [
            'json' => $payload,
        ], $retries);
    }

    /**
     * POST /pricing/prices-calculate
     * Pre-cálculo/validación antes del envío definitivo.
     */
    public function postPricesCalculate(array $payload, int $retries = 0): array
    {
        return $this->request('POST', 'pricing/prices-calculate', [
            'json' => $payload,
        ], $retries);
    }

    /**
     * GET /pricing/price-product-pdv?recId=&page=&size=
     * Obtiene precios actuales por PDV (paginado).
     */
    public function priceProductByPDV(string $recId, int $page = 0, int $size = 10, int $retries = 0): array
    {
        return $this->request('GET', 'pricing/price-product-pdv', [
            'query' => [
                'recId' => $recId,
                'page'  => $page,
                'size'  => $size,
            ],
        ], $retries);
    }

    /**
     * GET /pricing/price-lists
     * Listas/sublistas de PDV.
     */
    public function priceLists(int $retries = 0): array
    {
        return $this->request('GET', 'pricing/price-lists', [], $retries);
    }

    /**
     * GET /pricing/price-list-dealer-count?page=&size=
     * Conteo/paginado de precios cargados por plantilla.
     */
    public function priceListDealerCount(int $page = 0, int $size = 10, int $retries = 0): array
    {
        return $this->request('GET', 'pricing/price-list-dealer-count', [
            'query' => [
                'page' => $page,
                'size' => $size,
            ],
        ], $retries);
    }

    // ============================================================
    // product-controller
    // ============================================================

    /**
     * GET /pricing/search-products?searchValue=&page=&size=
     * Búsqueda de productos (paginada).
     */
    public function searchProducts(string $searchValue, int $page = 0, int $size = 10, int $retries = 0): array
    {
        return $this->request('GET', 'pricing/search-products', [
            'query' => [
                'searchValue' => $searchValue,
                'page'        => $page,
                'size'        => $size,
            ],
        ], $retries);
    }

    /**
     * Alias claro para paginación completa de productos (misma ruta).
     */
    public function searchProductsPaged(string $searchValue, int $page, int $size, int $retries = 0): array
    {
        return $this->searchProducts($searchValue, $page, $size, $retries);
    }
}
